<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Login de usuario
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $usuario = Usuario::where('email', $request->email)->first();

        if (!$usuario || !Hash::check($request->password, $usuario->password)) {
            throw ValidationException::withMessages([
                'email' => ['Las credenciales proporcionadas son incorrectas.'],
            ]);
        }

        // Verificar si el usuario está activo
        if ($usuario->estado !== 'activo') {
            return response()->json([
               'success' => false,
                'message' => 'Tu cuenta está inactiva. Contacta al administrador.',
            ], 403);
        }

        // Actualizar último acceso
        $usuario->actualizarUltimoAcceso();

        // Crear token
        $token = $usuario->createToken('auth-token')->plainTextToken;

        return response()->json([
            'success' => true,
            'message' => 'Login exitoso',
            'token' => $token,
            'user' => [
                'id' => $usuario->id,
                'nombre' => $usuario->nombre,
                'apellido' => $usuario->apellido,
                'nombre_completo' => $usuario->nombre_completo,
                'email' => $usuario->email,
                'rol' => $usuario->rol,
                'especialidad' => $usuario->especialidad,
                'foto_perfil' => $usuario->foto_perfil,
            ],
        ]);
    }

    /**
     * Logout de usuario
     */
    public function logout(Request $request)
    {
        // Eliminar el token actual
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logout exitoso',
        ]);
    }

    /**
     * Obtener usuario autenticado
     */
    public function user(Request $request)
    {
        $usuario = $request->user();

        if (!$usuario) {
            return response()->json([
                'success' => false,
                'message' => 'No autenticado'
            ], 401);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $usuario->id,
                'nombre' => $usuario->nombre,
                'apellido' => $usuario->apellido,
                'nombre_completo' => $usuario->nombre_completo,
                'email' => $usuario->email,
                'telefono' => $usuario->telefono,
                'horario' => $usuario->horario,
                'rol' => $usuario->rol,
                'especialidad' => $usuario->especialidad,
                'numero_licencia' => $usuario->numero_licencia,
                'foto_perfil' => $usuario->foto_perfil,
                'estado' => $usuario->estado,
            ],
        ]);
    }

    /**
     * Actualizar perfil usuario
     */
    public function updateProfile(Request $request)
    {
        $usuario = $request->user();

        $request->validate([
            'nombre' => 'sometimes|string|max:100',
            'apellido' => 'sometimes|string|max:100',
            'telefono' => 'sometimes|nullable|string|max:20',
            'horario' => 'sometimes|nullable|in:manana,tarde,24_7',
            'email' => 'sometimes|email|unique:usuarios,email,' . $usuario->id,
        ]);

        $usuario->update($request->only(['nombre', 'apellido', 'telefono', 'horario', 'email']));

        return response()->json([
            'success' => true,
            'message' => 'Perfil actualizado exitosamente',
            'data' => [
                'id' => $usuario->id,
                'nombre' => $usuario->nombre,
                'apellido' => $usuario->apellido,
                'nombre_completo' => $usuario->nombre_completo,
                'email' => $usuario->email,
                'telefono' => $usuario->telefono,
                'horario' => $usuario->horario,
                'rol' => $usuario->rol,
                'especialidad' => $usuario->especialidad,
                'foto_perfil' => $usuario->foto_perfil,
            ],
        ]);
    }

    /**
     * Cambiar contraseña
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ]);

        $usuario = $request->user();

        if (!Hash::check($request->current_password, $usuario->password)) {
            return response()->json([
                'success' => false,
                'message' => 'La contraseña actual es incorrecta',
            ], 400);
        }

        $usuario->password = Hash::make($request->new_password);
        $usuario->save();

        return response()->json([
            'success' => true,
            'message' => 'Contraseña cambiada exitosamente',
        ]);
    }
}
