<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Categoria;
use Illuminate\Http\Request;

class CategoriaController extends Controller
{
    public function index(Request $request)
    {
        $query = Categoria::query();

        if ($request->has("buscar")) {
            $buscar = $request->buscar;
            $query->where("nombre", "like", "%{$buscar}%");
        }

        if ($request->has("estado")) {
            $query->where("estado", $request->estado);
        }

        $query->orderBy("orden")->orderBy("nombre");

        $categorias = $query->get();

        // Asegurar codificación UTF-8 correcta
        $data = $categorias->map(function($item) {
            return [
                'id' => $item->id,
                'nombre' => mb_convert_encoding($item->nombre, 'UTF-8', 'UTF-8'),
                'descripcion' => $item->descripcion ? mb_convert_encoding($item->descripcion, 'UTF-8', 'UTF-8') : null,
                'icono' => $item->icono,
                'color' => $item->color,
                'estado' => $item->estado,
                'orden' => (int) $item->orden,
                'created_at' => $item->created_at,
                'updated_at' => $item->updated_at,
            ];
        });

        return response()->json([
            "success" => true,
            "data" => $data,
        ]);
    }

    public function store(Request $request)
    {
        if (auth()->user()->rol !== "admin") {
            return response()->json(["success" => false, "message" => "No tienes permisos"], 403);
        }

        $validated = $request->validate([
            "nombre" => "required|string|max:100|unique:categorias",
            "descripcion" => "nullable|string",
            "icono" => "nullable|string|max:50",
            "color" => "nullable|string|max:20",
            "estado" => "in:activo,inactivo",
            "orden" => "nullable|integer",
        ]);

        $validated["estado"] = $validated["estado"] ?? "activo";
        $validated["orden"] = $validated["orden"] ?? 0;

        $categoria = Categoria::create($validated);

        return response()->json([
            "success" => true,
            "message" => "Categoría creada exitosamente",
            "data" => $categoria,
        ], 201);
    }

    public function show($id)
    {
        $categoria = Categoria::findOrFail($id);
        return response()->json(["success" => true, "data" => $categoria]);
    }

    public function update(Request $request, $id)
    {
        if (auth()->user()->rol !== "admin") {
            return response()->json(["success" => false, "message" => "No tienes permisos"], 403);
        }

        $categoria = Categoria::findOrFail($id);

        $validated = $request->validate([
            "nombre" => "sometimes|string|max:100|unique:categorias,nombre," . $id,
            "descripcion" => "nullable|string",
            "icono" => "nullable|string|max:50",
            "color" => "nullable|string|max:20",
            "estado" => "sometimes|in:activo,inactivo",
            "orden" => "nullable|integer",
        ]);

        $categoria->update($validated);

        return response()->json([
            "success" => true,
            "message" => "Categoría actualizada",
            "data" => $categoria,
        ]);
    }

    public function destroy($id)
    {
        if (auth()->user()->rol !== "admin") {
            return response()->json(["success" => false, "message" => "No tienes permisos"], 403);
        }

        $categoria = Categoria::findOrFail($id);
        
        // Verificar si tiene estudios asociados
        $estudiosCount = \App\Models\Estudio::where("categoria", $categoria->nombre)->count();
        if ($estudiosCount > 0) {
            return response()->json([
                "success" => false, 
                "message" => "No se puede eliminar. Tiene " . $estudiosCount . " estudios asociados."
            ], 422);
        }

        $categoria->delete();
        return response()->json(["success" => true, "message" => "Categoría eliminada"]);
    }

    public function activas()
    {
        $categorias = Categoria::activas()->orderBy("orden")->orderBy("nombre")->get();
        
        // Asegurar codificación UTF-8 correcta
        $data = $categorias->map(function($item) {
            return [
                'id' => $item->id,
                'nombre' => mb_convert_encoding($item->nombre, 'UTF-8', 'UTF-8'),
                'descripcion' => $item->descripcion ? mb_convert_encoding($item->descripcion, 'UTF-8', 'UTF-8') : null,
                'icono' => $item->icono,
                'color' => $item->color,
                'estado' => $item->estado,
                'orden' => (int) $item->orden,
                'created_at' => $item->created_at,
                'updated_at' => $item->updated_at,
            ];
        });
        
        return response()->json(["success" => true, "data" => $data]);
    }
}
