<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Cita;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CitaController extends Controller
{
    /**
     * Listar citas
     */
    public function index(Request $request)
    {
        // Legacy fix: remove 'examen' relation as it does not exist
        $query = Cita::with(['paciente', 'medico', 'creadaPor']);

        // Filtrar por fecha
        if ($request->has('fecha')) {
            $query->whereDate('fecha_cita', $request->fecha);
        }

        // Filtrar por médico
        if ($request->has('medico_id')) {
            $query->where('medico_id', $request->medico_id);
        }

        // Filtrar por paciente
        if ($request->has('paciente_id')) {
            $query->where('paciente_id', $request->paciente_id);
        }

        // Filtrar por estado
        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        // Filtrar por pagado (Legacy incompatible)
        // if ($request->has('pagado')) {
        //     $query->where('pagado', filter_var($request->pagado, FILTER_VALIDATE_BOOLEAN));
        // }

        // Solo citas futuras
        if ($request->get('solo_futuras', false)) {
            $query->where('fecha_cita', '>=', now());
        }

        // Citas de hoy
        if ($request->get('hoy', false)) {
            $query->hoy();
        }

        // Ordenar - Las últimas citas creadas primero (DESC)
        $query->orderBy('fecha_creacion', 'desc');

        // Paginación
        $perPage = $request->get('per_page', 15);
        $citas = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $citas->items(),
            'pagination' => [
                'total' => $citas->total(),
                'per_page' => $citas->perPage(),
                'current_page' => $citas->currentPage(),
                'last_page' => $citas->lastPage(),
            ],
        ]);
    }

    /**
     * Crear nueva cita
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'paciente_id' => 'required|exists:pacientes,id',
            'medico_id' => 'required|exists:usuarios,id',
            'fecha_cita' => 'required|date',
            'duracion_minutos' => 'sometimes|integer|min:15|max:180',
            'tipo_cita' => 'required|in:consulta_general,consulta_especializada,seguimiento,emergencia,control',
            'motivo_consulta' => 'required|string',
            'sala_consultorio' => 'nullable|string|max:50',
            'observaciones' => 'nullable|string',
        ]);

        // Validar que la fecha no sea muy antigua (más de 1 año atrás)
        $fechaCita = Carbon::parse($validated['fecha_cita']);
        if ($fechaCita->lt(now()->subYear())) {
            return response()->json([
                'success' => false,
                'message' => 'La fecha de la cita no puede ser anterior a un año',
            ], 422);
        }

        // Establecer duración por defecto si no viene
        $validated['duracion_minutos'] = $validated['duracion_minutos'] ?? 30;

        // Verificar conflicto de horario
        if (Cita::tieneConflicto($validated['medico_id'], $validated['fecha_cita'], $validated['duracion_minutos'])) {
            return response()->json([
                'success' => false,
                'message' => 'El médico ya tiene una cita programada en ese horario',
            ], 422);
        }

        $validated['creada_por'] = $request->user()->id;
        $validated['estado'] = 'programada';
        $validated['pagado'] = false;

        $cita = Cita::create($validated);
        $cita->load(['paciente', 'medico', 'creadaPor']);

        return response()->json([
            'success' => true,
            'message' => 'Cita agendada exitosamente',
            'data' => $cita,
        ], 201);
    }

    /**
     * Mostrar una cita
     */
    public function show($id)
    {
        // Removed 'examen'
        $cita = Cita::with(['paciente', 'medico', 'creadaPor', 'historiaClinica'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $cita,
        ]);
    }

    /**
     * Actualizar cita
     */
    public function update(Request $request, $id)
    {
        $cita = Cita::findOrFail($id);

        $validated = $request->validate([
            'fecha_cita' => 'sometimes|date',
            'duracion_minutos' => 'sometimes|integer|min:15|max:180',
            'tipo_cita' => 'sometimes|in:consulta_general,consulta_especializada,seguimiento,emergencia,control',
            'motivo_consulta' => 'sometimes|string',
            'estado' => 'sometimes|in:programada,confirmada,en_curso,completada,cancelada,no_asistio',
            'sala_consultorio' => 'nullable|string|max:50',
            'observaciones' => 'nullable|string',
            // 'pagado' => 'sometimes|boolean',
        ]);

        // Si se actualiza fecha u hora, verificar conflicto
        if (isset($validated['fecha_cita']) || isset($validated['duracion_minutos'])) {
            $nuevaFecha = $validated['fecha_cita'] ?? $cita->fecha_cita;
            $nuevaDuracion = $validated['duracion_minutos'] ?? $cita->duracion_minutos;

            if (Cita::tieneConflicto($cita->medico_id, $nuevaFecha, $nuevaDuracion, $cita->id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'El médico ya tiene una cita programada en ese horario',
                ], 422);
            }
        }

        $cita->update($validated);
        $cita->load(['paciente', 'medico']);

        return response()->json([
            'success' => true,
            'message' => 'Cita actualizada exitosamente',
            'data' => $cita,
        ]);
    }

    /**
     * Cancelar cita
     */
    public function destroy($id)
    {
        $cita = Cita::findOrFail($id);
        
        $cita->update(['estado' => 'cancelada']);

        return response()->json([
            'success' => true,
            'message' => 'Cita cancelada exitosamente',
        ]);
    }

    /**
     * Obtener citas de hoy
     */
    public function hoy(Request $request)
    {
        $citas = Cita::with(['paciente', 'medico'])
            ->hoy()
            ->orderBy('fecha_cita', 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $citas,
        ]);
    }

    /**
     * Obtener agenda del médico
     */
    public function agenda(Request $request, $medicoId)
    {
        $fecha = $request->get('fecha', today());

        $citas = Cita::with(['paciente'])
            ->where('medico_id', $medicoId)
            ->whereDate('fecha_cita', $fecha)
            ->where('estado', '!=', 'cancelada')
            ->orderBy('fecha_cita', 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $citas,
        ]);
    }

    /**
     * Verificar disponibilidad de horario
     */
    public function verificarDisponibilidad(Request $request)
    {
        $request->validate([
            'medico_id' => 'required|exists:usuarios,id',
            'fecha_cita' => 'required|date',
            'duracion_minutos' => 'required|integer',
        ]);

        $disponible = !Cita::tieneConflicto(
            $request->medico_id,
            $request->fecha_cita,
            $request->duracion_minutos
        );

        return response()->json([
            'success' => true,
            'disponible' => $disponible,
        ]);
    }
}
