<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Estudio;
use Illuminate\Http\Request;

class EstudioController extends Controller
{
    private function encodeEstudio($item)
    {
        return [
            'id' => $item->id,
            'nombre' => utf8_encode($item->nombre),
            'descripcion' => $item->descripcion ? utf8_encode($item->descripcion) : null,
            'categoria' => utf8_encode($item->categoria),
            'codigo' => $item->codigo,
            'precio' => (float) $item->precio,
            'duracion_minutos' => (int) $item->duracion_minutos,
            'estado' => $item->estado,
            'created_at' => $item->created_at,
            'updated_at' => $item->updated_at,
        ];
    }

    public function index(Request $request)
    {
        $query = Estudio::query();

        if ($request->has("buscar")) {
            $buscar = $request->buscar;
            $query->where(function ($q) use ($buscar) {
                $q->where("nombre", "like", "%{$buscar}%")
                  ->orWhere("codigo", "like", "%{$buscar}%");
            });
        }

        if ($request->has("categoria")) {
            $query->where("categoria", $request->categoria);
        }

        if ($request->has("estado")) {
            $query->where("estado", $request->estado);
        }

        $query->orderBy("categoria")->orderBy("nombre");

        // Si es una petición simple (sin paginación), devolver todos los estudios
        if ($request->has('todos') || !$request->has('per_page')) {
            $estudios = $query->get();
            
            $data = [];
            foreach ($estudios as $item) {
                $data[] = $this->encodeEstudio($item);
            }
            
            return response()->json([
                "success" => true,
                "data" => $data,
            ]);
        }

        $estudios = $query->paginate($request->get("per_page", 15));
        
        $data = [];
        foreach ($estudios->items() as $item) {
            $data[] = $this->encodeEstudio($item);
        }

        return response()->json([
            "success" => true,
            "data" => $data,
            "pagination" => [
                "total" => $estudios->total(),
                "per_page" => $estudios->perPage(),
                "current_page" => $estudios->currentPage(),
                "last_page" => $estudios->lastPage(),
            ],
        ]);
    }

    public function store(Request $request)
    {
        if (!auth()->check() || auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos'], 403);
        }

        $validated = $request->validate([
            "nombre" => "required|string|max:200",
            "descripcion" => "nullable|string",
            "categoria" => "required|string|max:100",
            "codigo" => "required|string|max:20|unique:estudios",
            "precio" => "required|numeric|min:0",
            "duracion_minutos" => "nullable|integer|min:1",
            "estado" => "in:activo,inactivo",
        ]);

        $validated["estado"] = $validated["estado"] ?? "activo";
        $validated["duracion_minutos"] = $validated["duracion_minutos"] ?? 30;

        $estudio = Estudio::create($validated);

        return response()->json([
            "success" => true,
            "message" => "Estudio creado exitosamente",
            "data" => $estudio,
        ], 201);
    }

    public function show($id)
    {
        $estudio = Estudio::findOrFail($id);
        return response()->json(["success" => true, "data" => $estudio]);
    }

    public function update(Request $request, $id)
    {
        if (!auth()->check() || auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos'], 403);
        }

        $estudio = Estudio::findOrFail($id);

        $validated = $request->validate([
            "nombre" => "sometimes|string|max:200",
            "descripcion" => "nullable|string",
            "categoria" => "sometimes|string|max:100",
            "codigo" => "sometimes|string|max:20|unique:estudios,codigo," . $id,
            "precio" => "sometimes|numeric|min:0",
            "duracion_minutos" => "nullable|integer|min:1",
            "estado" => "sometimes|in:activo,inactivo",
        ]);

        $estudio->update($validated);

        return response()->json([
            "success" => true,
            "message" => "Estudio actualizado",
            "data" => $estudio,
        ]);
    }

    public function destroy($id)
    {
        if (!auth()->check() || auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos'], 403);
        }

        $estudio = Estudio::findOrFail($id);
        $estudio->delete();
        return response()->json(["success" => true, "message" => "Estudio eliminado"]);
    }

    public function categorias()
    {
        $categorias = Estudio::select("categoria")->distinct()->orderBy("categoria")->pluck("categoria");
        return response()->json(["success" => true, "data" => $categorias]);
    }

    public function porCategoria($categoria)
    {
        $estudios = Estudio::porCategoria($categoria)->activos()->orderBy("nombre")->get();
        return response()->json(["success" => true, "data" => $estudios]);
    }

    // SOLUCIÓN 3: Endpoint público para obtener estudios sin autenticación
    public function indexPublico(Request $request)
    {
        $query = Estudio::query();

        if ($request->has("estado")) {
            $query->where("estado", $request->estado);
        }

        $query->orderBy("categoria")->orderBy("nombre");
        
        $estudios = $query->get();
        
        $data = [];
        foreach ($estudios as $item) {
            $data[] = [
                'id' => $item->id,
                'nombre' => utf8_encode($item->nombre),
                'descripcion' => $item->descripcion ? utf8_encode($item->descripcion) : null,
                'categoria' => utf8_encode($item->categoria),
                'codigo' => $item->codigo,
                'precio' => (float) $item->precio,
                'duracion_minutos' => (int) $item->duracion_minutos,
                'estado' => $item->estado,
            ];
        }
        
        return response()->json([
            "success" => true,
            "data" => $data,
        ]);
    }
}
