<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\ExamenMedico;
use App\Models\PagoExamen;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ExamenController extends Controller
{
    /**
     * Listar exámenes
     */
    public function index(Request $request)
    {
        try {
            $query = ExamenMedico::with(['paciente', 'medicoSolicitante', 'pago']);

            // Filtros
            if ($request->has('paciente_id')) {
                $query->where('paciente_id', $request->paciente_id);
            }

            if ($request->has('estado')) {
                $query->where('estado', $request->estado);
            }

            if ($request->has('estado_pago')) {
                if ($request->estado_pago === 'pagado') {
                    $query->conPagoPagado();
                } elseif ($request->estado_pago === 'pendiente') {
                    $query->conPagoPendiente();
                }
            }

            $examenes = $query->orderBy('fecha_solicitud', 'desc')->get();

            $data = $examenes->map(function ($examen) {
                $pago = $examen->pago;
                return [
                    'id' => $examen->id,
                    'paciente_id' => $examen->paciente_id,
                    'paciente_nombre' => $examen->paciente ? 
                        $examen->paciente->nombre . ' ' . $examen->paciente->apellido : 'N/A',
                    'numero_historia' => $examen->paciente ? $examen->paciente->numero_historia : 'N/A',
                    'tipo_examen' => $examen->tipo_examen,
                    'descripcion' => $examen->descripcion,
                    'fecha_solicitud' => $examen->fecha_solicitud->format('Y-m-d'),
                    'fecha_realizacion' => $examen->fecha_realizacion ? 
                        $examen->fecha_realizacion->format('Y-m-d') : null,
                    'laboratorio_centro' => $examen->laboratorio_centro,
                    'medico_solicitante' => $examen->medicoSolicitante ? 
                        $examen->medicoSolicitante->nombre . ' ' . $examen->medicoSolicitante->apellido : 'N/A',
                    'estado' => $examen->estado,
                    'estado_formateado' => $examen->estado_formateado,
                    'tiene_resultado' => $examen->tieneResultado(),
                    'archivo_resultado' => $examen->archivo_resultado,
                    
                    // Información de pago
                    'pago' => $pago ? [
                        'id' => $pago->id,
                        'monto' => floatval($pago->monto),
                        'estado_pago' => $pago->estado_pago,
                        'estado_formateado' => $pago->estado_formateado,
                        'metodo_pago' => $pago->metodo_pago_formateado,
                        'fecha_pago' => $pago->fecha_pago ? $pago->fecha_pago->format('Y-m-d H:i') : null,
                    ] : null,
                    'estado_pago' => $pago ? $pago->estado_pago : 'sin_pago',
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $data,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al listar exámenes',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Crear examen
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'paciente_id' => 'required|exists:pacientes,id',
                'tipo_examen' => 'required|string|max:150',
                'descripcion' => 'nullable|string',
                'fecha_solicitud' => 'required|date',
                'laboratorio_centro' => 'nullable|string|max:200',
                'precio' => 'nullable|numeric|min:0',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Datos inválidos',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $examen = ExamenMedico::create([
                'paciente_id' => $request->paciente_id,
                'medico_solicitante_id' => auth()->id(),
                'historia_clinica_id' => $request->historia_clinica_id,
                'tipo_examen' => $request->tipo_examen,
                'descripcion' => $request->descripcion,
                'fecha_solicitud' => $request->fecha_solicitud,
                'laboratorio_centro' => $request->laboratorio_centro,
                'estado' => ExamenMedico::ESTADO_SOLICITADO,
            ]);

            // Crear pago si se especificó precio
            if ($request->filled('precio') && $request->precio > 0) {
                PagoExamen::create([
                    'examen_id' => $examen->id,
                    'paciente_id' => $request->paciente_id,
                    'monto' => $request->precio,
                    'metodo_pago' => 'efectivo',
                    'estado_pago' => PagoExamen::ESTADO_PENDIENTE,
                    'procesado_por' => auth()->id(),
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Examen creado exitosamente',
                'data' => $examen->load(['paciente', 'medicoSolicitante', 'pago']),
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear examen',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Ver detalle del examen
     */
    public function show($id)
    {
        try {
            $examen = ExamenMedico::with(['paciente', 'medicoSolicitante', 'pago'])->find($id);

            if (!$examen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Examen no encontrado',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $examen,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener examen',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Actualizar examen
     */
    public function update(Request $request, $id)
    {
        try {
            $examen = ExamenMedico::find($id);

            if (!$examen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Examen no encontrado',
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'tipo_examen' => 'sometimes|string|max:150',
                'descripcion' => 'nullable|string',
                'fecha_realizacion' => 'nullable|date',
                'laboratorio_centro' => 'nullable|string|max:200',
                'resultados' => 'nullable|string',
                'observaciones' => 'nullable|string',
                'estado' => 'sometimes|in:solicitado,en_proceso,completado,cancelado',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Datos inválidos',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $examen->update($request->only([
                'tipo_examen',
                'descripcion',
                'fecha_realizacion',
                'laboratorio_centro',
                'resultados',
                'observaciones',
                'estado',
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Examen actualizado exitosamente',
                'data' => $examen,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar examen',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Subir resultado del examen
     */
    public function subirResultado(Request $request, $id)
    {
        try {
            $examen = ExamenMedico::with('pago')->find($id);

            if (!$examen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Examen no encontrado',
                ], 404);
            }

            // Verificar que el examen esté pagado
            if (!$examen->estaPagado()) {
                return response()->json([
                    'success' => false,
                    'message' => 'El examen debe estar pagado antes de subir el resultado',
                ], 403);
            }

            $validator = Validator::make($request->all(), [
                'archivo' => 'required|file|mimes:pdf,jpg,jpeg,png,doc,docx|max:10240', // 10MB max
            ], [
                'archivo.required' => 'Debe seleccionar un archivo',
                'archivo.mimes' => 'El archivo debe ser PDF, JPG, PNG, DOC o DOCX',
                'archivo.max' => 'El archivo no debe superar los 10MB',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Archivo inválido',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Eliminar archivo anterior si existe
            if ($examen->archivo_resultado) {
                Storage::disk('public')->delete('examenes/' . $examen->archivo_resultado);
            }

            // Guardar nuevo archivo
            $archivo = $request->file('archivo');
            $nombreArchivo = Str::slug($examen->tipo_examen) . '_' . 
                             $examen->paciente_id . '_' . 
                             time() . '.' . 
                             $archivo->getClientOriginalExtension();

            $ruta = $archivo->storeAs('examenes', $nombreArchivo, 'public');

            // Actualizar examen
            $examen->update([
                'archivo_resultado' => $nombreArchivo,
                'estado' => ExamenMedico::ESTADO_COMPLETADO,
                'fecha_realizacion' => $examen->fecha_realizacion ?? now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Resultado subido exitosamente',
                'data' => [
                    'archivo' => $nombreArchivo,
                    'url' => asset('storage/examenes/' . $nombreArchivo),
                ],
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al subir resultado',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Eliminar examen
     */
    public function destroy($id)
    {
        try {
            $examen = ExamenMedico::find($id);

            if (!$examen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Examen no encontrado',
                ], 404);
            }

            // Eliminar archivo si existe
            if ($examen->archivo_resultado) {
                Storage::disk('public')->delete('examenes/' . $examen->archivo_resultado);
            }

            $examen->delete();

            return response()->json([
                'success' => true,
                'message' => 'Examen eliminado exitosamente',
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar examen',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
