<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\HistoriaClinica;
use App\Models\HistoriaEvolucion;
use App\Models\HistoriaArchivo;
use App\Models\Cie10Catalog;
use App\Models\RecetaMedica;
use App\Models\MedicamentoReceta;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class HistoriaClinicaController extends Controller
{
    /**
     * Obtener historial completo de un paciente
     */
    public function index(Request $request)
    {
        $pacienteId = $request->query('paciente_id');

        if (!$pacienteId) {
            return response()->json(['error' => 'paciente_id requerido'], 400);
        }

        $historias = HistoriaClinica::with(['medico:id,nombre,apellido,especialidad', 'cie10:id,codigo,descripcion', 'archivos', 'recetasMedicas.medicamentos', 'examenesMedicos'])
            ->where('paciente_id', $pacienteId)
            ->orderBy('fecha_consulta', 'desc')
            ->paginate(10);

        return response()->json($historias);
    }

    /**
     * Crear una nueva historia clínica (SOAP + CIE-10)
     */
    public function store(Request $request)
    {
        $request->validate([
            'cita_id' => 'required|exists:citas,id|unique:historias_clinicas,cita_id',
            'paciente_id' => 'required|exists:pacientes,id',
            'medico_id' => 'required|exists:usuarios,id',
            'motivo_consulta' => 'required|string',
            'cie10_id' => 'nullable|exists:cie10_catalog,id',
            'tipo_diagnostico' => 'in:presuntivo,definitivo',
            // Validaciones JSON
            'notas_soap' => 'nullable|array',
            'notas_soap.subjetivo' => 'nullable|string',
            'notas_soap.objetivo' => 'nullable|string',
            'notas_soap.analisis' => 'nullable|string',
            'notas_soap.plan' => 'nullable|string',
            'presion_arterial' => 'nullable|string',
            'temperatura' => 'nullable|numeric',
            'peso' => 'nullable|numeric',
        ]);

        DB::beginTransaction();
        try {
            $data = $request->all();
            
            // Copiar código para búsqueda rápida si existe CIE-10
            if ($request->has('cie10_id')) {
                $cie10 = Cie10Catalog::find($request->cie10_id);
                $data['cie10_codigo_texto'] = $cie10->codigo ?? null;
                
                // Incrementar contador de uso
                if ($cie10) $cie10->increment('frecuencia_uso');
            }

            // Datos de auditoría
            $data['auditoria'] = [
                'creado_por' => auth()->id() ?? $request->medico_id,
                'creado_fecha' => now()->toIso8601String(),
                'ip' => $request->ip()
            ];

            $historia = HistoriaClinica::create($data);

            // Manejar archivos si se envían en la creación
            if ($request->hasFile('archivos')) {
                $this->subirArchivos($historia, $request->file('archivos'));
            }

            // Actualizar estado de la cita a 'completada'
            DB::table('citas')->where('id', $request->cita_id)->update(['estado' => 'completada']);

            // ---------------------------------------------------------
            // Procesar Receta Médica Integrada
            // ---------------------------------------------------------
            if ($request->has('receta_medicamentos') && is_array($request->receta_medicamentos)) {
                // Filtrar medicamentos vacíos
                $medicamentos = array_filter($request->receta_medicamentos, function ($m) {
                    return !empty($m['nombre_medicamento']); // Solo si tiene nombre
                });

                if (count($medicamentos) > 0) {
                    $receta = RecetaMedica::create([
                        'historia_clinica_id' => $historia->id,
                        'paciente_id' => $request->paciente_id,
                        'medico_id' => auth()->id() ?? $request->medico_id,
                        'fecha_emision' => now(),
                        'vigencia_dias' => 30,
                        'instrucciones_generales' => $request->receta_instrucciones ?? null,
                        'estado' => 'activa'
                    ]);

                    foreach ($medicamentos as $med) {
                        try {
                            MedicamentoReceta::create([
                                'receta_id' => $receta->id,
                                'nombre_medicamento' => $med['nombre_medicamento'],
                                'dosis' => $med['dosis'] ?? 'N/A',
                                'via_administracion' => $med['via_administracion'] ?? 'oral',
                                'frecuencia' => $med['frecuencia'] ?? 'Cada 8 horas',
                                'duracion_dias' => $med['duracion_dias'] ?? 1,
                                'cantidad_total' => $med['cantidad_total'] ?? 1,
                                'instrucciones_especiales' => $med['instrucciones_especiales'] ?? null,
                            ]);
                        } catch (\Exception $eM) {
                            Log::warning("Error guardando medicamento: " . $eM->getMessage());
                            // Continuar con los siguientes
                        }
                    }
                }
            }
            // ---------------------------------------------------------
            // Procesar Exámenes Médicos (Laboratorio/Imagen)
            // ---------------------------------------------------------
            if ($request->has('ordenes_laboratorio') && is_array($request->ordenes_laboratorio)) {
                foreach ($request->ordenes_laboratorio as $ordenLab) {
                    try {
                        \App\Models\ExamenMedico::create([
                            'paciente_id' => $request->paciente_id,
                            'medico_solicitante_id' => auth()->id() ?? $request->medico_id,
                            'historia_clinica_id' => $historia->id,
                            'tipo_examen' => str_contains(strtolower($ordenLab), 'radiografía') ? 'imagenologia' : 'laboratorio',
                            'descripcion' => $ordenLab,
                            'fecha_solicitud' => now(),
                            'estado' => \App\Models\ExamenMedico::ESTADO_SOLICITADO
                        ]);
                    } catch (\Exception $eE) {
                        Log::warning("Error guardando orden de lab: " . $eE->getMessage());
                    }
                }
            }
            // ---------------------------------------------------------

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Historia clínica y receta guardadas exitosamente',
                'data' => $historia->load('cie10')
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creando historia: ' . $e->getMessage());
            return response()->json(['error' => 'Error al guardar historia: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Mostrar detalle completo de una historia
     */
    public function show($id)
    {
        $historia = HistoriaClinica::with([
            'medico:id,nombre,apellido,especialidad', 
            'cie10', 
            'evoluciones.medico:id,nombre,apellido', 
            'archivos',
            'recetasMedicas.medicamentos',
            'examenesMedicos'
        ])->findOrFail($id);

        return response()->json($historia);
    }

    /**
     * Agregar una evolución a una historia existente
     */
    public function agregarEvolucion(Request $request, $id)
    {
        $historia = HistoriaClinica::findOrFail($id);

        $request->validate([
            'notas_soap' => 'required|array',
            'medico_id' => 'required|exists:usuarios,id',
        ]);

        $evolucion = HistoriaEvolucion::create([
            'historia_clinica_id' => $historia->id,
            'medico_id' => $request->medico_id,
            'fecha_evolucion' => now(),
            'notas_soap' => $request->notas_soap,
            'estado_paciente' => $request->estado_paciente ?? 'estable',
            'cie10_id' => $request->cie10_id,
            'observaciones' => $request->observaciones
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Evolución agregada',
            'data' => $evolucion
        ]);
    }

    /**
     * Subir archivos adjuntos (Rayos X, Labs, etc)
     */
    public function subirArchivosAdjuntos(Request $request, $id)
    {
        $historia = HistoriaClinica::findOrFail($id);

        if (!$request->hasFile('archivos')) {
            return response()->json(['error' => 'No se enviaron archivos'], 400);
        }

        $archivosGuardados = $this->subirArchivos($historia, $request->file('archivos'));

        return response()->json([
            'success' => true,
            'message' => count($archivosGuardados) . ' archivos subidos',
            'data' => $archivosGuardados
        ]);
    }

    /**
     * Helper para subir archivos
     */
    private function subirArchivos(HistoriaClinica $historia, $archivos)
    {
        $guardados = [];
        if (!is_array($archivos)) {
            $archivos = [$archivos];
        }

        foreach ($archivos as $archivo) {
            $nombreOriginal = $archivo->getClientOriginalName();
            $extension = $archivo->getClientOriginalExtension();
            $tipo = $this->detectarTipoArchivo($extension);
            
            // Guardar en storage/app/public/historias/{id}
            $path = $archivo->storeAs(
                "public/historias/{$historia->id}", 
                uniqid() . '_' . $nombreOriginal
            );
            
            // Ruta relativa para guardar en BD (sin 'public/')
            $rutaRelativa = str_replace('public/', '', $path);

            $nuevoArchivo = HistoriaArchivo::create([
                'historia_clinica_id' => $historia->id,
                'tipo' => $tipo,
                'nombre_original' => $nombreOriginal,
                'ruta_archivo' => $rutaRelativa, // historia/1/xyz.jpg
                'mime_type' => $archivo->getMimeType(),
                'tamano_bytes' => $archivo->getSize(),
                'subido_por' => auth()->id()
            ]);

            $guardados[] = $nuevoArchivo;
        }

        return $guardados;
    }

    private function detectarTipoArchivo($ext)
    {
        $ext = strtolower($ext);
        if (in_array($ext, ['jpg', 'jpeg', 'png', 'webp'])) return 'otro'; // Podría ser foto herida
        if (in_array($ext, ['pdf'])) return 'laboratorio'; 
        if (in_array($ext, ['dcm'])) return 'rayos_x';
        return 'otro';
    }

    /**
     * Búsqueda de CIE-10 para autocompletado
     */
    public function buscarCie10(Request $request)
    {
        $query = $request->query('q');
        
        if (!$query) {
            return response()->json(Cie10Catalog::populares(20)->get());
        }

        $resultados = Cie10Catalog::buscar($query)
            ->orderByDesc('frecuencia_uso')
            ->limit(20)
            ->get();

        return response()->json($resultados);
    }
}
