<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Paciente;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PacienteController extends Controller
{
    public function index(Request $request)
    {
        $query = Paciente::query(); // Modelo Legacy no tiene 'examen' relation por default asi que quitamos with

        if ($request->has('buscar') && $request->buscar) {
            $termino = $request->buscar;
            $query->where(function ($q) use ($termino) {
                $q->where('nombre', 'like', "%{$termino}%")
                  ->orWhere('apellido', 'like', "%{$termino}%")
                  ->orWhere('numero_documento', 'like', "%{$termino}%")
                  ->orWhere('telefono', 'like', "%{$termino}%");
            });
        }

        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        // Estado pago ya no es core de paciente en legacy, pero lo dejaremos opcional
        if ($request->has('estado_pago')) {
            // $query->where('estado_pago', $request->estado_pago); 
        }

        $query->orderBy('fecha_registro', 'desc'); // Legacy usa fecha_registro

        $pacientes = $query->paginate($request->get('per_page', 15));

        $data = array_map(function($p) {
            return $this->encodePaciente($p);
        }, $pacientes->items());

        return response()->json([
            'success' => true,
            'data' => $data,
            'pagination' => [
                'total' => $pacientes->total(),
                'per_page' => $pacientes->perPage(),
                'current_page' => $pacientes->currentPage(),
                'last_page' => $pacientes->lastPage(),
            ],
        ]);
    }

    private function encodePaciente($paciente)
    {
        return [
            'id' => (int) $paciente->id,
            // Campos Legacy reales
            'numero_documento' => (string) ($paciente->numero_documento ?? ''),
            'telefono'         => (string) ($paciente->telefono ?? ''),
            'numero_historia'  => (string) ($paciente->numero_historia ?? ''),
            'nombre'           => (string) ($paciente->nombre ?? ''),
            'apellido'         => (string) ($paciente->apellido ?? ''),
            'sexo'             => (string) ($paciente->sexo ?? ''),
            'direccion'        => $paciente->direccion ? (string) $paciente->direccion : null,
            // fecha_nacimiento es Carbon con cast 'date'
            'fecha_nacimiento' => $paciente->fecha_nacimiento
                ? $paciente->fecha_nacimiento->format('Y-m-d')
                : null,
            'email'            => $paciente->email ? (string) $paciente->email : null,
            'notas'            => $paciente->observaciones ? (string) $paciente->observaciones : null,
            'estado'           => (string) ($paciente->estado ?? 'activo'),
            // Nuevos campos fase 3
            'alergias'         => (string) ($paciente->alergias ?? ''),
            'grupo_sanguineo'  => (string) ($paciente->grupo_sanguineo ?? 'Desconocido'),
            'edad'             => $paciente->edad ?? ($paciente->fecha_nacimiento ? $paciente->fecha_nacimiento->age : 0),
            'foto_paciente'    => $paciente->foto_paciente ? (string) $paciente->foto_paciente : null,
            'contacto_emergencia'=> $paciente->contacto_emergencia ? (string) $paciente->contacto_emergencia : null,
            'telefono_emergencia'=> $paciente->telefono_emergencia ? (string) $paciente->telefono_emergencia : null,
            
            // created_at puede ser Carbon (fecha_registro) o null
            'created_at'       => $paciente->created_at
                ? $paciente->created_at->toDateTimeString()
                : null,

            // Alias de compatibilidad con frontend antiguo
            'ci'          => (string) ($paciente->numero_documento ?? ''),
            'celular'     => (string) ($paciente->telefono ?? ''),
            'examen_id'   => null,
            'precio'      => 0,
            'metodo_pago' => 'Efectivo',
            'estado_pago' => 'pendiente',
            'examen'      => null,
        ];
    }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'numero_documento' => 'required|string|max:50|unique:pacientes',
            'nombre' => 'required|string|max:100',
            'apellido' => 'required|string|max:100',
            'telefono' => 'nullable|string|max:20',
            'sexo' => 'required|in:M,F,Otro,Masculino,Femenino', // Aceptamos ambos sets
            'direccion' => 'nullable|string',
            'fecha_nacimiento' => 'nullable|date',
            'email' => 'nullable|email|max:150',
            'notas' => 'nullable|string',
            'estado' => 'in:activo,inactivo',
            // Campos opcionales alias
            'ci' => 'nullable|string',
            'celular' => 'nullable|string',
        ]);

        // Normalizar datos alias
        if (empty($validated['numero_documento']) && !empty($request->ci)) {
            $validated['numero_documento'] = $request->ci;
        }
        if (empty($validated['telefono']) && !empty($request->celular)) {
            $validated['telefono'] = $request->celular;
        }

        // Normalización Sexo
        if ($validated['sexo'] == 'Masculino') $validated['sexo'] = 'M';
        if ($validated['sexo'] == 'Femenino') $validated['sexo'] = 'F';

        // Generar numero_historia
        $validated['numero_historia'] = 'HC-' . $validated['numero_documento'];
        $validated['estado'] = $validated['estado'] ?? 'activo';

        // Mapear notas a observaciones
        $dataToCreate = [
            'numero_documento' => $validated['numero_documento'],
            'numero_historia' => $validated['numero_historia'],
            'nombre' => $validated['nombre'],
            'apellido' => $validated['apellido'],
            'telefono' => $validated['telefono'],
            'sexo' => $validated['sexo'],
            'direccion' => $validated['direccion'],
            'fecha_nacimiento' => $validated['fecha_nacimiento'],
            'email' => $validated['email'],
            'observaciones' => $validated['notas'],
            'estado' => $validated['estado'],
            'registrado_por' => auth()->id()
        ];

        $paciente = Paciente::create($dataToCreate);

        return response()->json([
            'success' => true,
            'message' => 'Paciente registrado exitosamente',
            'data' => $this->encodePaciente($paciente),
        ], 201);
    }

    public function show($id)
    {
        $paciente = Paciente::findOrFail($id);
        return response()->json(['success' => true, 'data' => $this->encodePaciente($paciente)]);
    }

    public function update(Request $request, $id)
    {
        $paciente = Paciente::findOrFail($id);

        $validated = $request->validate([
            'numero_documento' => 'sometimes|string|max:50|unique:pacientes,numero_documento,' . $id,
            'nombre' => 'sometimes|string|max:100',
            'apellido' => 'sometimes|string|max:100',
            'telefono' => 'nullable|string|max:20',
            'sexo' => 'sometimes|in:M,F,Otro,Masculino,Femenino',
            'direccion' => 'nullable|string',
            'fecha_nacimiento' => 'nullable|date',
            'email' => 'nullable|email|max:150',
            'notas' => 'nullable|string',
            'estado' => 'sometimes|in:activo,inactivo',
            // Alias update
            'ci' => 'sometimes|string',
            'celular' => 'sometimes|string',
        ]);

        $dataToUpdate = [];
        if ($request->has('numero_documento')) $dataToUpdate['numero_documento'] = $request->numero_documento;
        if ($request->has('ci')) $dataToUpdate['numero_documento'] = $request->ci;
        
        if ($request->has('telefono')) $dataToUpdate['telefono'] = $request->telefono;
        if ($request->has('celular')) $dataToUpdate['telefono'] = $request->celular;
        
        if ($request->has('nombre')) $dataToUpdate['nombre'] = $request->nombre;
        if ($request->has('apellido')) $dataToUpdate['apellido'] = $request->apellido;
        
        if ($request->has('sexo')) {
             $s = $request->sexo;
             if ($s == 'Masculino') $s = 'M';
             if ($s == 'Femenino') $s = 'F';
             $dataToUpdate['sexo'] = $s;
        }
        
        if ($request->has('direccion')) $dataToUpdate['direccion'] = $request->direccion;
        if ($request->has('fecha_nacimiento')) $dataToUpdate['fecha_nacimiento'] = $request->fecha_nacimiento;
        if ($request->has('email')) $dataToUpdate['email'] = $request->email;
        if ($request->has('notas')) $dataToUpdate['observaciones'] = $request->notas;
        if ($request->has('estado')) $dataToUpdate['estado'] = $request->estado;

        $paciente->update($dataToUpdate);

        return response()->json([
            'success' => true,
            'message' => 'Paciente actualizado',
            'data' => $this->encodePaciente($paciente),
        ]);
    }

    public function destroy($id)
    {
        $paciente = Paciente::findOrFail($id);
        $paciente->delete(); // Si hay SoftDeletes en modelo bien, sino delete.
        return response()->json(['success' => true, 'message' => 'Paciente eliminado']);
    }

    public function cambiarEstadoPago($id)
    {
        // Legacy NO tiene estado_pago. Retornamos success simulado.
        return response()->json([
            'success' => true,
            'message' => 'Estado de pago no soportado en versión actual',
            'estado_pago' => 'pagado',
        ]);
    }
}
