<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\PagoExamen;
use App\Models\ExamenMedico;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class PagoExamenController extends Controller
{
    /**
     * Listar todos los pagos
     */
    public function index(Request $request)
    {
        try {
            $query = PagoExamen::with(['examen', 'paciente', 'procesador']);

            // Filtros
            if ($request->has('estado_pago')) {
                $query->where('estado_pago', $request->estado_pago);
            }

            if ($request->has('paciente_id')) {
                $query->where('paciente_id', $request->paciente_id);
            }

            $pagos = $query->orderBy('fecha_creacion', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $pagos,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al listar pagos',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Registrar un pago para un examen
     */
    public function registrarPago(Request $request, $examenId)
    {
        try {
            // Validar entrada
            $validator = Validator::make($request->all(), [
                'monto' => 'required|numeric|min:0',
                'metodo_pago' => 'required|in:efectivo,tarjeta,transferencia,seguro,otro',
                'numero_comprobante' => 'nullable|string|max:100',
                'observaciones' => 'nullable|string',
            ], [
                'monto.required' => 'El monto es obligatorio',
                'monto.numeric' => 'El monto debe ser un número',
                'monto.min' => 'El monto debe ser mayor o igual a 0',
                'metodo_pago.required' => 'El método de pago es obligatorio',
                'metodo_pago.in' => 'Método de pago no válido',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Datos inválidos',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Buscar el examen
            $examen = ExamenMedico::with('paciente')->find($examenId);

            if (!$examen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Examen no encontrado',
                ], 404);
            }

            DB::beginTransaction();

            try {
                // Verificar si ya existe un pago
                $pago = PagoExamen::where('examen_id', $examenId)->first();

                if ($pago) {
                    // Actualizar pago existente
                    $pago->update([
                        'monto' => $request->monto,
                        'metodo_pago' => $request->metodo_pago,
                        'estado_pago' => PagoExamen::ESTADO_PAGADO,
                        'fecha_pago' => now(),
                        'numero_comprobante' => $request->numero_comprobante,
                        'observaciones' => $request->observaciones,
                        'procesado_por' => auth()->id(),
                    ]);

                    $mensaje = 'Pago actualizado exitosamente';
                } else {
                    // Crear nuevo pago
                    $pago = PagoExamen::create([
                        'examen_id' => $examenId,
                        'paciente_id' => $examen->paciente_id,
                        'monto' => $request->monto,
                        'metodo_pago' => $request->metodo_pago,
                        'estado_pago' => PagoExamen::ESTADO_PAGADO,
                        'fecha_pago' => now(),
                        'numero_comprobante' => $request->numero_comprobante,
                        'observaciones' => $request->observaciones,
                        'procesado_por' => auth()->id(),
                    ]);

                    $mensaje = 'Pago registrado exitosamente';
                }

                // Actualizar estado del examen si está en solicitado
                if ($examen->estado === ExamenMedico::ESTADO_SOLICITADO) {
                    $examen->update([
                        'estado' => ExamenMedico::ESTADO_EN_PROCESO,
                    ]);
                }

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => $mensaje,
                    'data' => $pago->load(['examen', 'paciente', 'procesador']),
                ], 200);

            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al registrar pago',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Actualizar estado de pago
     */
    public function actualizarEstado(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'estado_pago' => 'required|in:pendiente,pagado,parcial,cancelado,reembolsado',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Datos inválidos',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $pago = PagoExamen::find($id);

            if (!$pago) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pago no encontrado',
                ], 404);
            }

            $pago->update([
                'estado_pago' => $request->estado_pago,
                'procesado_por' => auth()->id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Estado de pago actualizado',
                'data' => $pago,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar estado',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Ver detalle de un pago
     */
    public function show($id)
    {
        try {
            $pago = PagoExamen::with(['examen', 'paciente', 'procesador'])->find($id);

            if (!$pago) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pago no encontrado',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $pago,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener pago',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Eliminar un pago
     */
    public function destroy($id)
    {
        try {
            $pago = PagoExamen::find($id);

            if (!$pago) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pago no encontrado',
                ], 404);
            }

            $pago->delete();

            return response()->json([
                'success' => true,
                'message' => 'Pago eliminado exitosamente',
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar pago',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
