<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\RecetaMedica;
use App\Models\MedicamentoReceta;
use App\Models\MedicamentoCatalogo;
use App\Models\Paciente;
use App\Models\HistoriaClinica;
use Illuminate\Http\Request;
use Carbon\Carbon;

class RecetaMedicaController extends Controller
{
    public function index(Request $request)
    {
        $query = RecetaMedica::with(['paciente', 'medico', 'medicamentos', 'historiaClinica']);

        if ($request->has('paciente_id')) {
            $query->where('paciente_id', $request->paciente_id);
        }

        if ($request->has('medico_id')) {
            $query->where('medico_id', $request->medico_id);
        }

        if ($request->has('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('fecha_emision', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('fecha_emision', '<=', $request->fecha_hasta);
        }

        $recetas = $query->orderBy('fecha_emision', 'desc')->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => $recetas->items(),
            'pagination' => [
                'total' => $recetas->total(),
                'per_page' => $recetas->perPage(),
                'current_page' => $recetas->currentPage(),
                'last_page' => $recetas->lastPage(),
            ],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'paciente_id' => 'required|exists:pacientes,id',
            'medico_id' => 'required|exists:usuarios,id',
            'historia_clinica_id' => 'nullable|exists:historias_clinicas,id',
            'fecha_emision' => 'required|date',
            'vigencia_dias' => 'required|integer|min:1|max:365',
            'instrucciones_generales' => 'nullable|string',
            'estado' => 'in:activa,cancelada,vencida',
            'medicamentos' => 'required|array|min:1',
            'medicamentos.*.nombre_medicamento' => 'required|string',
            'medicamentos.*.dosis' => 'required|string',
            'medicamentos.*.via_administracion' => 'required|string',
            'medicamentos.*.frecuencia' => 'required|string',
            'medicamentos.*.duracion_dias' => 'required|integer|min:1',
            'medicamentos.*.cantidad_total' => 'required|integer|min:1',
            'medicamentos.*.instrucciones_especiales' => 'nullable|string',
        ]);

        $receta = RecetaMedica::create([
            'paciente_id' => $validated['paciente_id'],
            'medico_id' => $validated['medico_id'],
            'historia_clinica_id' => $validated['historia_clinica_id'] ?? null,
            'fecha_emision' => $validated['fecha_emision'],
            'vigencia_dias' => $validated['vigencia_dias'],
            'instrucciones_generales' => $validated['instrucciones_generales'] ?? null,
            'estado' => $validated['estado'] ?? 'activa',
        ]);

        foreach ($validated['medicamentos'] as $medicamento) {
            $receta->medicamentos()->create([
                'nombre_medicamento' => $medicamento['nombre_medicamento'],
                'dosis' => $medicamento['dosis'],
                'via_administracion' => $medicamento['via_administracion'],
                'frecuencia' => $medicamento['frecuencia'],
                'duracion_dias' => $medicamento['duracion_dias'],
                'cantidad_total' => $medicamento['cantidad_total'],
                'instrucciones_especiales' => $medicamento['instrucciones_especiales'] ?? null,
            ]);
        }

        $receta->load(['paciente', 'medico', 'medicamentos', 'historiaClinica']);

        return response()->json([
            'success' => true,
            'message' => 'Receta médica creada exitosamente',
            'data' => $receta,
        ], 201);
    }

    public function show($id)
    {
        $receta = RecetaMedica::with(['paciente', 'medico', 'medicamentos', 'historiaClinica'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $receta,
        ]);
    }

    public function update(Request $request, $id)
    {
        $receta = RecetaMedica::findOrFail($id);

        $validated = $request->validate([
            'historia_clinica_id' => 'nullable|exists:historias_clinicas,id',
            'fecha_emision' => 'sometimes|date',
            'vigencia_dias' => 'sometimes|integer|min:1|max:365',
            'instrucciones_generales' => 'nullable|string',
            'estado' => 'sometimes|in:activa,cancelada,vencida',
        ]);

        $receta->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Receta actualizada exitosamente',
            'data' => $receta->fresh(['paciente', 'medico', 'medicamentos', 'historiaClinica']),
        ]);
    }

    public function destroy($id)
    {
        $receta = RecetaMedica::findOrFail($id);
        
        $receta->medicamentos()->delete();
        $receta->delete();

        return response()->json([
            'success' => true,
            'message' => 'Receta eliminada exitosamente',
        ]);
    }

    public function porPaciente($pacienteId)
    {
        $recetas = RecetaMedica::with(['medico', 'medicamentos'])
            ->where('paciente_id', $pacienteId)
            ->orderBy('fecha_emision', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $recetas,
        ]);
    }

    public function catalogo(Request $request)
    {
        $query = MedicamentoCatalogo::activos();

        if ($request->has('categoria')) {
            $query->where('categoria', $request->categoria);
        }

        if ($request->has('buscar')) {
            $buscar = $request->buscar;
            $query->where(function ($q) use ($buscar) {
                $q->where('nombre_generico', 'like', "%{$buscar}%")
                  ->orWhere('nombre_comercial', 'like', "%{$buscar}%");
            });
        }

        $medicamentos = $query->orderBy('nombre_generico')->paginate($request->get('per_page', 50));

        return response()->json([
            'success' => true,
            'data' => $medicamentos->items(),
            'pagination' => [
                'total' => $medicamentos->total(),
                'per_page' => $medicamentos->perPage(),
                'current_page' => $medicamentos->currentPage(),
                'last_page' => $medicamentos->lastPage(),
            ],
        ]);
    }

    public function buscar(Request $request)
    {
        $request->validate([
            'q' => 'required|string|min:2'
        ]);

        $medicamentos = MedicamentoCatalogo::activos()
            ->where(function ($query) use ($request) {
                $query->where('nombre_generico', 'like', "%{$request->q}%")
                      ->orWhere('nombre_comercial', 'like', "%{$request->q}%");
            })
            ->limit(20)
            ->get(['id', 'nombre_generico', 'nombre_comercial', 'concentracion', 'categoria', 'precio_bs', 'via_administracion', 'presentacion', 'alternativas', 'dosis_adulto']);

        return response()->json([
            'success' => true,
            'data' => $medicamentos,
        ]);
    }

    public function porCategoria($categoria)
    {
        $medicamentos = MedicamentoCatalogo::activos()
            ->where('categoria', $categoria)
            ->orderBy('nombre_generico')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $medicamentos,
        ]);
    }

    public function categorias()
    {
        return response()->json([
            'success' => true,
            'data' => MedicamentoCatalogo::getCategorias(),
        ]);
    }
}
