<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Paciente;
use App\Models\Resultado;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ResultadoController extends Controller
{
    public function index(Request $request)
    {
        $query = Resultado::with(['paciente', 'examen', 'subidoPor']);

        if ($request->has('paciente_id')) {
            $query->where('paciente_id', $request->paciente_id);
        }

        if ($request->has('examen_id')) {
            $query->where('examen_id', $request->examen_id);
        }

        $query->orderBy('subido_en', 'desc');

        $resultados = $query->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => $resultados->items(),
            'pagination' => [
                'total' => $resultados->total(),
                'per_page' => $resultados->perPage(),
                'current_page' => $resultados->currentPage(),
                'last_page' => $resultados->lastPage(),
            ],
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'paciente_id' => 'required|exists:pacientes,id',
            'archivo' => 'required|file|mimes:pdf,jpg,jpeg,webp|max:10240',
            'observaciones' => 'nullable|string',
        ]);

        $paciente = Paciente::findOrFail($validated['paciente_id']);

        if ($paciente->estado_pago !== 'pagado') {
            return response()->json([
                'success' => false,
                'message' => 'No se pueden subir resultados hasta que el paciente pague'
            ], 403);
        }

        $archivo = $request->file('archivo');
        $nombreOriginal = $archivo->getClientOriginalName();
        $extension = $archivo->getClientOriginalExtension();
        $tipoArchivo = $this->getTipoArchivo($extension);
        $tamano = $archivo->getSize();

        $nombreArchivo = time() . '_' . str_replace(' ', '_', $nombreOriginal);
        $ruta = $archivo->storeAs('resultados', $nombreArchivo, 'public');

        $resultado = Resultado::create([
            'paciente_id' => $validated['paciente_id'],
            'examen_id' => $paciente->examen_id,
            'archivo_nombre' => $nombreOriginal,
            'archivo_ruta' => $ruta,
            'archivo_tipo' => $tipoArchivo,
            'archivo_tamano' => $tamano,
            'observaciones' => $validated['observaciones'] ?? null,
            'subido_por' => auth()->id(),
            'subido_en' => now(),
            'estado' => 'activo',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Resultado subido exitosamente',
            'data' => $resultado->load(['paciente', 'examen', 'subidoPor']),
        ], 201);
    }

    public function show($id)
    {
        $resultado = Resultado::with(['paciente', 'examen', 'subidoPor'])->findOrFail($id);
        return response()->json(['success' => true, 'data' => $resultado]);
    }

    public function download($id)
    {
        $resultado = Resultado::with('paciente')->findOrFail($id);

        // Verificar que el paciente ha pagado
        if ($resultado->paciente->estado_pago !== 'pagado') {
            return response()->json([
                'success' => false,
                'message' => 'El examen no está disponible hasta que se realice el pago'
            ], 403);
        }

        if (!Storage::disk('public')->exists($resultado->archivo_ruta)) {
            return response()->json([
                'success' => false,
                'message' => 'Archivo no encontrado'
            ], 404);
        }

        return Storage::disk('public')->download(
            $resultado->archivo_ruta,
            $resultado->archivo_nombre
        );
    }

    public function view($id)
    {
        $resultado = Resultado::with('paciente')->findOrFail($id);

        // Verificar que el paciente ha pagado
        if ($resultado->paciente->estado_pago !== 'pagado') {
            return response()->json([
                'success' => false,
                'message' => 'El examen no está disponible hasta que se realice el pago'
            ], 403);
        }

        if (!Storage::disk('public')->exists($resultado->archivo_ruta)) {
            return response()->json([
                'success' => false,
                'message' => 'Archivo no encontrado'
            ], 404);
        }

        $path = Storage::disk('public')->path($resultado->archivo_ruta);
        $mimeType = $resultado->archivo_tipo;

        return response()->file($path, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . $resultado->archivo_nombre . '"'
        ]);
    }

    public function destroy($id)
    {
        $resultado = Resultado::findOrFail($id);

        if (Storage::disk('public')->exists($resultado->archivo_ruta)) {
            Storage::disk('public')->delete($resultado->archivo_ruta);
        }

        $resultado->delete();

        return response()->json([
            'success' => true,
            'message' => 'Resultado eliminado'
        ]);
    }

    public function porPaciente($pacienteId)
    {
        $resultados = Resultado::with(['examen', 'subidoPor'])
            ->where('paciente_id', $pacienteId)
            ->orderBy('subido_en', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $resultados
        ]);
    }

    private function getTipoArchivo($extension)
    {
        $tipos = [
            'pdf' => 'application/pdf',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'webp' => 'image/webp',
        ];

        return $tipos[strtolower($extension)] ?? 'application/octet-stream';
    }
}
