<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Usuario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UsuarioController extends Controller
{
    public function index(Request $request)
    {
        $query = Usuario::query();

        if ($request->has("buscar")) {
            $buscar = $request->buscar;
            $query->where(function ($q) use ($buscar) {
                $q->where("nombre", "like", "%{$buscar}%")
                  ->orWhere("apellido", "like", "%{$buscar}%")
                  ->orWhere("email", "like", "%{$buscar}%");
            });
        }

        if ($request->has("rol")) {
            $query->where("rol", $request->rol);
        }

        if ($request->has("estado")) {
            $query->where("estado", $request->estado);
        }

        $query->orderBy($request->get("ordenar", "nombre"), $request->get("direccion", "asc"));

        $usuarios = $query->paginate($request->get("per_page", 15));

        return response()->json([
            "success" => true,
            "data" => $usuarios->items(),
            "pagination" => [
                "total" => $usuarios->total(),
                "per_page" => $usuarios->perPage(),
                "current_page" => $usuarios->currentPage(),
                "last_page" => $usuarios->lastPage(),
            ],
        ]);
    }

    public function store(Request $request)
    {
        // Solo admin puede crear usuarios
        if (auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos para crear usuarios'], 403);
        }

        $validated = $request->validate([
            "nombre" => "required|string|max:100",
            "apellido" => "required|string|max:100",
            "email" => "required|email|unique:usuarios",
            "telefono" => "nullable|string|max:20",
            "rol" => "required|in:admin,medico,enfermero,recepcionista",
            "horario" => "required|in:manana,tarde",
            "password" => "required|min:8",
        ]);

        $validated["password"] = Hash::make($validated["password"]);
        $validated["estado"] = "activo";

        $usuario = Usuario::create($validated);

        return response()->json([
            "success" => true,
            "message" => "Usuario creado exitosamente",
            "data" => $usuario,
        ], 201);
    }

    public function show($id)
    {
        $usuario = Usuario::findOrFail($id);
        return response()->json(["success" => true, "data" => $usuario]);
    }

    public function update(Request $request, $id)
    {
        // Solo admin puede actualizar usuarios
        if (auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos para editar usuarios'], 403);
        }

        $usuario = Usuario::findOrFail($id);

        $validated = $request->validate([
            "nombre" => "sometimes|string|max:100",
            "apellido" => "sometimes|string|max:100",
            "email" => "sometimes|email|unique:usuarios,email," . $id,
            "telefono" => "nullable|string|max:20",
            "rol" => "sometimes|in:admin,medico,enfermero,recepcionista",
            "horario" => "sometimes|in:manana,tarde,24_7",
            "estado" => "sometimes|in:activo,inactivo,suspendido",
        ]);

        $usuario->update($validated);

        return response()->json([
            "success" => true,
            "message" => "Usuario actualizado",
            "data" => $usuario,
        ]);
    }

    public function destroy($id)
    {
        // Solo admin puede eliminar usuarios
        if (auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos para eliminar usuarios'], 403);
        }

        $usuario = Usuario::findOrFail($id);
        
        if ($usuario->id === auth()->id()) {
            return response()->json(["success" => false, "message" => "No puedes eliminarte a ti mismo"], 422);
        }

        $usuario->delete();
        return response()->json(["success" => true, "message" => "Usuario eliminado"]);
    }

    public function cambiarEstado(Request $request, $id)
    {
        // Solo admin puede cambiar estado
        if (auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos para cambiar el estado'], 403);
        }

        $usuario = Usuario::findOrFail($id);
        
        $request->validate(["estado" => "required|in:activo,inactivo,suspendido"]);
        
        $usuario->update(["estado" => $request->estado]);
        
        return response()->json(["success" => true, "message" => "Estado actualizado", "data" => $usuario]);
    }

    // Admin puede cambiar contraseña de otros usuarios
    public function cambiarPassword(Request $request, $id)
    {
        // Solo admin puede cambiar contraseña de otros
        if (auth()->user()->rol !== 'admin') {
            return response()->json(['success' => false, 'message' => 'No tienes permisos'], 403);
        }

        $usuario = Usuario::findOrFail($id);

        $request->validate([
            'password' => 'required|min:8|confirmed',
        ]);

        $usuario->update(['password' => Hash::make($request->password)]);

        return response()->json(['success' => true, 'message' => 'Contraseña actualizada correctamente']);
    }
}
