<?php

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Models\Paciente;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PacienteAuthController extends Controller
{
    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'ci' => 'required|string',
                'celular' => 'required|string',
            ], [
                'ci.required' => 'La cédula de identidad es obligatoria',
                'celular.required' => 'El número de celular es obligatorio',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Datos inválidos',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $paciente = Paciente::where('numero_documento', $request->ci)->first();

            if (!$paciente) {
                return response()->json([
                    'success' => false,
                    'message' => 'Credenciales incorrectas. Verifica tu C.I.',
                ], 401);
            }

            // La contraseña del paciente es su número de teléfono
            if ($paciente->telefono !== $request->celular) {
                return response()->json([
                    'success' => false,
                    'message' => 'Credenciales incorrectas. Verifica tu número de celular.',
                ], 401);
            }

            if ($paciente->estado !== 'activo') {
                return response()->json([
                    'success' => false,
                    'message' => 'Cuenta inactiva. Contacte al administrador.',
                ], 403);
            }

            $token = 'patient_' . bin2hex(random_bytes(32));

            return response()->json([
                'success' => true,
                'message' => 'Login exitoso',
                'token' => $token,
                'paciente' => [
                    'id' => $paciente->id,
                    'ci' => $paciente->numero_documento,
                    'numero_historia' => $paciente->numero_historia ?? 'N/A',
                    'nombre_completo' => $paciente->nombre . ' ' . $paciente->apellido,
                    'nombre' => $paciente->nombre,
                    'apellido' => $paciente->apellido,
                    'email' => $paciente->email,
                    'celular' => $paciente->telefono,
                    'fecha_nacimiento' => $paciente->fecha_nacimiento,
                    'sexo' => $paciente->sexo,
                    'direccion' => $paciente->direccion,
                ],
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error en el servidor',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function logout(Request $request)
    {
        return response()->json([
            'success' => true,
            'message' => 'Logout exitoso',
        ], 200);
    }

    public function me(Request $request)
    {
        try {
            $paciente = $request->user();

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $paciente->id,
                    'ci' => $paciente->numero_documento,
                    'numero_historia' => $paciente->numero_historia ?? 'N/A',
                    'nombre_completo' => $paciente->nombre . ' ' . $paciente->apellido,
                    'nombre' => $paciente->nombre,
                    'apellido' => $paciente->apellido,
                    'email' => $paciente->email,
                    'celular' => $paciente->telefono,
                    'fecha_nacimiento' => $paciente->fecha_nacimiento,
                    'sexo' => $paciente->sexo,
                    'direccion' => $paciente->direccion,
                    'estado' => $paciente->estado,
                    'created_at' => $paciente->created_at,
                ],
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener datos',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
