<?php

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Models\Cita;
use App\Models\HistoriaClinica;
use App\Models\ExamenMedico;
use Illuminate\Http\Request;

class PacienteDashboardController extends Controller
{
    public function index(Request $request)
    {
        try {
            $paciente = $request->user();
            $pacienteId = $paciente->id;

            // Citas del paciente
            $totalCitas = Cita::where('paciente_id', $pacienteId)->count();
            $citasProximas = Cita::where('paciente_id', $pacienteId)
                ->whereIn('estado', ['programada', 'confirmada'])
                ->whereDate('fecha_cita', '>=', now())
                ->count();

            // Historias clínicas
            $totalHistorias = HistoriaClinica::where('paciente_id', $pacienteId)->count();

            // Exámenes del paciente
            $totalExamenes = ExamenMedico::where('paciente_id', $pacienteId)->count();
            $examenesPendientes = ExamenMedico::where('paciente_id', $pacienteId)
                ->where('estado', 'solicitado')->count();

            // Próximas citas (hasta 3)
            $proximasCitas = Cita::where('paciente_id', $pacienteId)
                ->whereIn('estado', ['programada', 'confirmada'])
                ->whereDate('fecha_cita', '>=', now())
                ->with(['medico:id,nombre,apellido,especialidad'])
                ->orderBy('fecha_cita')
                ->limit(3)
                ->get()
                ->map(fn($c) => [
                    'id' => $c->id,
                    'medico' => 'Dr. ' . ($c->medico->nombre ?? '') . ' ' . ($c->medico->apellido ?? ''),
                    'especialidad' => $c->medico->especialidad ?? 'Medicina General',
                    'fecha_formateada' => \Carbon\Carbon::parse($c->fecha_cita)->format('d/m/Y H:i'),
                    'tipo_cita' => $c->tipo_cita,
                    'estado' => $c->estado,
                ]);

            // Últimos exámenes (hasta 5)
            $ultimosExamenes = ExamenMedico::where('paciente_id', $pacienteId)
                ->with(['medicoSolicitante:id,nombre,apellido'])
                ->orderBy('fecha_solicitud', 'desc')
                ->limit(5)
                ->get()
                ->map(fn($e) => [
                    'id' => $e->id,
                    'tipo_examen' => $e->descripcion,
                    'fecha_formateada' => \Carbon\Carbon::parse($e->fecha_solicitud)->format('d/m/Y'),
                    'medico' => ($e->medicoSolicitante->nombre ?? '') . ' ' . ($e->medicoSolicitante->apellido ?? ''),
                    'estado_examen' => $e->estado,
                    'estado_pago' => 'pagado',
                    'puede_ver' => $e->estado === 'completado',
                ]);

            // Último diagnóstico (historia más reciente)
            $ultimaHistoria = HistoriaClinica::where('paciente_id', $pacienteId)
                ->with(['cie10:id,codigo,descripcion', 'medico:id,nombre,apellido'])
                ->orderBy('fecha_consulta', 'desc')
                ->first();

            $data = [
                'paciente_nombre' => $paciente->nombre . ' ' . $paciente->apellido,
                'resumen' => [
                    'total_citas' => $totalCitas,
                    'citas_proximas' => $citasProximas,
                    'total_examenes' => $totalExamenes,
                    'total_historias' => $totalHistorias,
                    'examenes_pendientes_pago' => $examenesPendientes,
                ],
                'proximas_citas' => $proximasCitas,
                'ultimos_examenes' => $ultimosExamenes,
                'ultima_historia' => $ultimaHistoria ? [
                    'fecha' => \Carbon\Carbon::parse($ultimaHistoria->fecha_consulta)->format('d/m/Y'),
                    'motivo' => $ultimaHistoria->motivo_consulta,
                    'diagnostico' => $ultimaHistoria->cie10 ? $ultimaHistoria->cie10->codigo . ' - ' . $ultimaHistoria->cie10->descripcion : ($ultimaHistoria->notas_soap['analisis'] ?? 'Sin diagnóstico registrado'),
                    'medico' => 'Dr. ' . ($ultimaHistoria->medico->nombre ?? '') . ' ' . ($ultimaHistoria->medico->apellido ?? ''),
                ] : null,
            ];

            return response()->json([
                'success' => true,
                'data' => $data,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar dashboard: ' . $e->getMessage(),
            ], 500);
        }
    }
}
