<?php

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Models\ExamenMedico;
use Illuminate\Http\Request;

class PacienteExamenController extends Controller
{
    public function index(Request $request)
    {
        try {
            $paciente = $request->user();
            $estado = $request->input('estado', 'todos');

            $query = ExamenMedico::where('paciente_id', $paciente->id)
                ->with(['medicoSolicitante:id,nombre,apellido,especialidad'])
                ->orderBy('fecha_solicitud', 'desc');

            if ($estado === 'completados') {
                $query->where('estado', 'completado');
            } elseif ($estado === 'pendientes') {
                $query->whereIn('estado', ['solicitado', 'en_proceso']);
            }

            $examenes = $query->get()->map(fn($e) => [
                'id' => $e->id,
                'tipo_examen' => $e->tipo_examen,
                'descripcion' => $e->descripcion,
                'fecha_solicitud' => $e->fecha_solicitud,
                'fecha_formateada' => \Carbon\Carbon::parse($e->fecha_solicitud)->format('d/m/Y'),
                'laboratorio_centro' => $e->laboratorio_centro ?? 'Cibertronia Salud',
                'medico_solicitante' => 'Dr. ' . ($e->medicoSolicitante->nombre ?? '') . ' ' . ($e->medicoSolicitante->apellido ?? ''),
                'estado_examen' => $e->estado,
                'estado_examen_formateado' => match($e->estado) {
                    'solicitado' => 'Solicitado',
                    'en_proceso' => 'En Proceso',
                    'completado' => 'Completado',
                    'cancelado' => 'Cancelado',
                    default => ucfirst($e->estado),
                },
                'estado_pago' => 'pagado',
                'tiene_resultado' => !empty($e->archivo_resultado),
                'puede_ver_resultado' => $e->estado === 'completado',
                'archivo_resultado' => $e->archivo_resultado,
                'resultados' => $e->resultados,
                'observaciones' => $e->observaciones,
            ]);

            return response()->json([
                'success' => true,
                'data' => $examenes,
                'total' => $examenes->count(),
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar exámenes: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function show(Request $request, $id)
    {
        try {
            $paciente = $request->user();
            $examen = ExamenMedico::where('id', $id)
                ->where('paciente_id', $paciente->id)
                ->with('medicoSolicitante:id,nombre,apellido,especialidad')
                ->firstOrFail();

            return response()->json([
                'success' => true,
                'data' => $examen,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Examen no encontrado',
            ], 404);
        }
    }

    public function descargar(Request $request, $id)
    {
        return response()->json(['success' => false, 'message' => 'Descarga no disponible en este momento'], 404);
    }

    public function ver(Request $request, $id)
    {
        return response()->json(['success' => false, 'message' => 'Visualización no disponible en este momento'], 404);
    }
}
