<?php

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Models\HistoriaClinica;
use Illuminate\Http\Request;
use Carbon\Carbon;

class PacienteHistoriaController extends Controller
{
    public function index(Request $request)
    {
        try {
            $paciente = $request->user();

            $historias = HistoriaClinica::where('paciente_id', $paciente->id)
                ->with([
                    'medico:id,nombre,apellido,especialidad',
                    'cie10:id,codigo,descripcion',
                    'recetasMedicas.medicamentos',
                    'examenesMedicos',
                ])
                ->orderBy('fecha_consulta', 'desc')
                ->get()
                ->map(fn($h) => [
                    'id'                   => $h->id,
                    'fecha_consulta'       => $h->fecha_consulta,
                    'fecha_formateada'     => Carbon::parse($h->fecha_consulta)->format('d/m/Y'),
                    'motivo_consulta'      => $h->motivo_consulta,
                    'diagnostico'          => $h->diagnostico,
                    'tratamiento'          => $h->tratamiento,
                    'recomendaciones'      => $h->recomendaciones,
                    'proxima_cita'         => $h->proxima_cita,
                    // Signos vitales
                    'presion_arterial'     => $h->presion_arterial,
                    'frecuencia_cardiaca'  => $h->frecuencia_cardiaca,
                    'temperatura'          => $h->temperatura,
                    'peso'                 => $h->peso,
                    'altura'               => $h->altura,
                    'saturacion_oxigeno'   => $h->saturacion_oxigeno,
                    // SOAP
                    'notas_soap'           => $h->notas_soap,
                    // Relaciones
                    'medico'               => $h->medico ? [
                        'nombre'      => 'Dr. ' . $h->medico->nombre . ' ' . $h->medico->apellido,
                        'especialidad'=> $h->medico->especialidad ?? 'Medicina General',
                    ] : null,
                    'cie10'       => $h->cie10 ? $h->cie10->codigo . ' - ' . $h->cie10->descripcion : null,
                    'tipo_diagnostico' => $h->tipo_diagnostico,
                    'examenes'    => $h->examenesMedicos ? $h->examenesMedicos->map(fn($e) => [
                        'id'          => $e->id,
                        'descripcion' => $e->descripcion,
                        'tipo'        => $e->tipo_examen,
                        'estado'      => $e->estado,
                        'resultados'  => $e->resultados,
                        'fecha'       => $e->fecha_solicitud ? Carbon::parse($e->fecha_solicitud)->format('d/m/Y') : null,
                    ]) : [],
                    'recetas'     => $h->recetasMedicas ? $h->recetasMedicas->map(fn($r) => [
                        'id'                     => $r->id,
                        'fecha_emision'          => Carbon::parse($r->fecha_emision)->format('d/m/Y'),
                        'instrucciones_generales'=> $r->instrucciones_generales,
                        'medicamentos'           => $r->medicamentos ? $r->medicamentos->map(fn($m) => [
                            'nombre'     => $m->nombre_medicamento,
                            'dosis'      => $m->dosis,
                            'frecuencia' => $m->frecuencia,
                            'duracion'   => $m->duracion_dias . ' días',
                        ]) : [],
                    ]) : [],
                ]);

            return response()->json([
                'success' => true,
                'data'    => $historias,
                'total'   => $historias->count(),
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al cargar historial: ' . $e->getMessage(),
            ], 500);
        }
    }
}
