<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Cita extends Model
{
    use HasFactory;

    protected $table = 'citas';

    protected $fillable = [
        'paciente_id',
        'medico_id',
        'fecha_cita',
        'duracion_minutos',
        'tipo_cita',
        'motivo_consulta',
        'estado',
        'pagado',
        'sala_consultorio',
        'observaciones',
        'creada_por',
    ];

    protected $casts = [
        'fecha_cita' => 'datetime',
        'pagado' => 'boolean',
    ];

    const CREATED_AT = 'fecha_creacion';
    const UPDATED_AT = 'fecha_actualizacion';

    // =========================================================
    // SCOPES
    // =========================================================

    /** Citas de hoy */
    public function scopeHoy($query)
    {
        return $query->whereDate('fecha_cita', today());
    }

    /** Citas programadas (no canceladas) */
    public function scopeProgramadas($query)
    {
        return $query->where('estado', 'programada');
    }

    /** Citas completadas */
    public function scopeCompletadas($query)
    {
        return $query->where('estado', 'completada');
    }

    // =========================================================
    // MÉTODOS ESTÁTICOS
    // =========================================================

    /**
     * Verifica si el médico tiene conflicto en ese horario.
     */
    public static function tieneConflicto($medicoId, $fechaCita, $duracionMinutos, $exceptoId = null)
    {
        $inicio = Carbon::parse($fechaCita);
        $fin = $inicio->copy()->addMinutes($duracionMinutos);

        $query = self::where('medico_id', $medicoId)
            ->whereNotIn('estado', ['cancelada', 'no_asistio'])
            ->where(function ($q) use ($inicio, $fin) {
                // La nueva cita empieza dentro de una existente
                $q->where(function ($q2) use ($inicio, $fin) {
                    $q2->where('fecha_cita', '<', $fin)
                       ->whereRaw("DATE_ADD(fecha_cita, INTERVAL duracion_minutos MINUTE) > ?", [$inicio]);
                });
            });

        if ($exceptoId) {
            $query->where('id', '!=', $exceptoId);
        }

        return $query->exists();
    }

    // =========================================================
    // RELACIONES
    // =========================================================
    public function paciente()
    {
        return $this->belongsTo(Paciente::class, 'paciente_id');
    }

    public function medico()
    {
        return $this->belongsTo(Usuario::class, 'medico_id');
    }

    public function creadaPor()
    {
        return $this->belongsTo(Usuario::class, 'creada_por');
    }

    public function historiaClinica()
    {
        return $this->hasOne(HistoriaClinica::class, 'cita_id');
    }
}
