<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ExamenMedico extends Model
{
    use HasFactory;

    protected $table = 'examenes_medicos';

    protected $fillable = [
        'paciente_id',
        'medico_solicitante_id',
        'historia_clinica_id',
        'tipo_examen',
        'descripcion',
        'fecha_solicitud',
        'fecha_realizacion',
        'laboratorio_centro',
        'resultados',
        'archivo_resultado',
        'estado',
        'observaciones',
    ];

    protected $casts = [
        'fecha_solicitud' => 'date',
        'fecha_realizacion' => 'date',
        'fecha_creacion' => 'datetime',
        'fecha_actualizacion' => 'datetime',
    ];

    protected $dates = [
        'fecha_solicitud',
        'fecha_realizacion',
        'fecha_creacion',
        'fecha_actualizacion',
    ];

    // Constantes para estados
    const ESTADO_SOLICITADO = 'solicitado';
    const ESTADO_EN_PROCESO = 'en_proceso';
    const ESTADO_COMPLETADO = 'completado';
    const ESTADO_CANCELADO = 'cancelado';

    /**
     * Relación con el paciente
     */
    public function paciente()
    {
        return $this->belongsTo(Paciente::class, 'paciente_id');
    }

    /**
     * Relación con el médico solicitante
     */
    public function medicoSolicitante()
    {
        return $this->belongsTo(Usuario::class, 'medico_solicitante_id');
    }

    /**
     * Relación con la historia clínica
     */
    public function historiaClinica()
    {
        return $this->belongsTo(HistoriaClinica::class, 'historia_clinica_id');
    }

    /**
     * Relación con el pago (uno a uno)
     */
    public function pago()
    {
        return $this->hasOne(PagoExamen::class, 'examen_id');
    }

    /**
     * Scopes
     */
    public function scopeSolicitados($query)
    {
        return $query->where('estado', self::ESTADO_SOLICITADO);
    }

    public function scopeEnProceso($query)
    {
        return $query->where('estado', self::ESTADO_EN_PROCESO);
    }

    public function scopeCompletados($query)
    {
        return $query->where('estado', self::ESTADO_COMPLETADO);
    }

    public function scopePorPaciente($query, $pacienteId)
    {
        return $query->where('paciente_id', $pacienteId);
    }

    public function scopeConPagoPendiente($query)
    {
        return $query->whereHas('pago', function ($q) {
            $q->where('estado_pago', PagoExamen::ESTADO_PENDIENTE);
        });
    }

    public function scopeConPagoPagado($query)
    {
        return $query->whereHas('pago', function ($q) {
            $q->where('estado_pago', PagoExamen::ESTADO_PAGADO);
        });
    }

    /**
     * Verificar si el examen tiene pago
     */
    public function tienePago(): bool
    {
        return $this->pago()->exists();
    }

    /**
     * Verificar si el examen está pagado
     */
    public function estaPagado(): bool
    {
        return $this->tienePago() && $this->pago->isPagado();
    }

    /**
     * Verificar si tiene resultado cargado
     */
    public function tieneResultado(): bool
    {
        return !empty($this->archivo_resultado);
    }

    /**
     * Verificar si el paciente puede ver el resultado
     */
    public function puedeVerResultado(): bool
    {
        return $this->estaPagado() && $this->tieneResultado();
    }

    /**
     * Obtener el estado formateado
     */
    public function getEstadoFormateadoAttribute(): string
    {
        $estados = [
            self::ESTADO_SOLICITADO => 'Solicitado',
            self::ESTADO_EN_PROCESO => 'En Proceso',
            self::ESTADO_COMPLETADO => 'Completado',
            self::ESTADO_CANCELADO => 'Cancelado',
        ];

        return $estados[$this->estado] ?? $this->estado;
    }

    /**
     * Obtener la URL del archivo de resultado
     */
    public function getUrlResultadoAttribute(): ?string
    {
        if (!$this->archivo_resultado) {
            return null;
        }

        return asset('storage/examenes/' . $this->archivo_resultado);
    }

    /**
     * Marcar como completado
     */
    public function marcarComoCompletado(): bool
    {
        $this->estado = self::ESTADO_COMPLETADO;
        $this->fecha_realizacion = now();
        return $this->save();
    }
}
