<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\Cie10Catalog;
use App\Models\HistoriaEvolucion;
use App\Models\HistoriaArchivo;

class HistoriaClinica extends Model
{
    use HasFactory;

    protected $table = 'historias_clinicas';

    protected $fillable = [
        'cita_id',
        'paciente_id',
        'medico_id',
        'fecha_consulta',
        'presion_arterial',
        'frecuencia_cardiaca',
        'temperatura',
        'frecuencia_respiratoria',
        'peso',
        'altura',
        'saturacion_oxigeno',
        'motivo_consulta',
        'sintomas',
        'diagnostico',
        'tratamiento',
        'observaciones',
        'examenes_solicitados',
        'medicamentos_recetados',
        'proxima_cita',
        'recomendaciones',
        'archivos_adjuntos', // Legacy field, kept for backward compatibility
        // Campos Nuevos Fase 2
        'cie10_id',
        'cie10_codigo_texto',
        'tipo_diagnostico',
        'notas_soap',
        'examen_fisico',
        'bloqueada_para_edicion',
        'fecha_bloqueo',
        'auditoria'
    ];

    protected $casts = [
        'fecha_consulta' => 'datetime',
        'proxima_cita' => 'date',
        'archivos_adjuntos' => 'array',
        'fecha_creacion' => 'datetime',
        'fecha_actualizacion' => 'datetime',
        'temperatura' => 'decimal:2',
        'peso' => 'decimal:2',
        'altura' => 'decimal:2',
        // Nuevos Casts
        'notas_soap' => 'array',
        'examen_fisico' => 'array',
        'auditoria' => 'array',
        'bloqueada_para_edicion' => 'boolean',
        'fecha_bloqueo' => 'datetime'
    ];

    const CREATED_AT = 'fecha_creacion';
    const UPDATED_AT = 'fecha_actualizacion';

    /**
     * Relaciones
     */
    public function cita()
    {
        return $this->belongsTo(Cita::class, 'cita_id');
    }

    public function paciente()
    {
        return $this->belongsTo(Paciente::class, 'paciente_id');
    }

    public function medico()
    {
        return $this->belongsTo(Usuario::class, 'medico_id');
    }

    public function recetasMedicas()
    {
        return $this->hasMany(RecetaMedica::class, 'historia_clinica_id');
    }

    public function examenesMedicos()
    {
        return $this->hasMany(ExamenMedico::class, 'historia_clinica_id');
    }

    /**
     * Nuevas Relaciones Fase 2
     */
    public function cie10()
    {
        return $this->belongsTo(Cie10Catalog::class, 'cie10_id');
    }

    public function evoluciones()
    {
        return $this->hasMany(HistoriaEvolucion::class, 'historia_clinica_id')->orderBy('fecha_evolucion', 'desc');
    }

    public function archivos()
    {
        return $this->hasMany(HistoriaArchivo::class, 'historia_clinica_id');
    }

    /**
     * Accesor para IMC (calculado automáticamente)
     */
    public function getImcAttribute()
    {
        if ($this->peso && $this->altura) {
            $alturaMetros = $this->altura / 100;
            return round($this->peso / ($alturaMetros * $alturaMetros), 2);
        }
        return null;
    }

    /**
     * Accesor para clasificación IMC
     */
    public function getClasificacionImcAttribute()
    {
        $imc = $this->imc;
        
        if (!$imc) return null;
        
        if ($imc < 18.5) return 'Bajo peso';
        if ($imc < 25) return 'Peso normal';
        if ($imc < 30) return 'Sobrepeso';
        if ($imc < 35) return 'Obesidad I';
        if ($imc < 40) return 'Obesidad II';
        return 'Obesidad III';
    }

    /**
     * Scopes
     */
    public function scopeDelPaciente($query, $pacienteId)
    {
        return $query->where('paciente_id', $pacienteId);
    }

    public function scopeDelMedico($query, $medicoId)
    {
        return $query->where('medico_id', $medicoId);
    }

    public function scopeRecientes($query, $limite = 10)
    {
        return $query->orderBy('fecha_consulta', 'desc')->limit($limite);
    }
}
