<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MedicamentoCatalogo extends Model
{
    use HasFactory;

    protected $table = 'medicamentos_catalogo';

    protected $fillable = [
        'nombre_generico',
        'nombre_comercial',
        'concentracion',
        'via_administracion',
        'categoria',
        'precio_bs',
        'presentacion',
        'alternativas',
        'contraindicaciones',
        'dosis_adulto',
        'dosis_pediatrica',
        'estado',
    ];

    protected $casts = [
        'precio_bs' => 'decimal:2',
        'alternativas' => 'array',
        'estado' => 'string',
    ];

    public function scopeActivos($query)
    {
        return $query->where('estado', 'activo');
    }

    public function scopePorCategoria($query, $categoria)
    {
        return $query->where('categoria', $categoria);
    }

    public function scopeBuscar($query, $termino)
    {
        return $query->where(function ($q) use ($termino) {
            $q->where('nombre_generico', 'like', "%{$termino}%")
              ->orWhere('nombre_comercial', 'like', "%{$termino}%");
        });
    }

    public function getAlternativasListAttribute()
    {
        return $this->alternativas ?? [];
    }

    public static function getCategorias()
    {
        return [
            'cardiovascular' => 'Cardiovascular',
            'analgesico' => 'Analgésicos/Antiinflamatorios',
            'antibiotico' => 'Antibióticos',
            'respiratorio' => 'Respiratorios',
            'diabetes' => 'Diabetes',
            'gastrointestinal' => 'Gastrointestinales',
            'psiquiatrico' => 'Psiquiátricos/Neurológicos',
            'antialergico' => 'Antialérgicos',
            'vitaminas' => 'Vitaminas/Suplementos',
            'otros' => 'Otros',
        ];
    }

    public static function getViasAdministracion()
    {
        return [
            'oral' => 'Oral',
            'iv' => 'Intravenosa (IV)',
            'im' => 'Intramuscular (IM)',
            'sc' => 'Subcutánea (SC)',
            'topical' => 'Tópica',
            'nasal' => 'Nasal',
            'ocular' => 'Ocular',
            'auricular' => 'Auricular',
            'inhalatoria' => 'Inhalatoria',
            'rectal' => 'Rectal',
            'sublingual' => 'Sublingual',
        ];
    }
}
