<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Paciente extends Model
{
    protected $table = 'pacientes';

    protected $fillable = [
        'numero_historia',
        'nombre',
        'apellido',
        'fecha_nacimiento',
        'sexo', // M, F, Otro
        'tipo_documento', // DNI, Pasaporte, etc
        'numero_documento',
        'telefono',
        'telefono_emergencia',
        'email',
        'direccion',
        'ciudad',
        'ocupacion',
        'estado_civil',
        'grupo_sanguineo',
        'alergias',
        'observaciones',
        'foto_paciente',
        'estado',
        'registrado_por'
    ];

    protected $casts = [
        'fecha_nacimiento' => 'date',
    ];

    const CREATED_AT = 'fecha_registro';
    const UPDATED_AT = 'fecha_actualizacion';

    // =========================================================
    // SCOPES
    // =========================================================

    /** Pacientes activos */
    public function scopeActivos($query)
    {
        return $query->where('estado', 'activo');
    }

    /** Pacientes inactivos */
    public function scopeInactivos($query)
    {
        return $query->where('estado', 'inactivo');
    }

    /**
     * Scope 'pendientes' — Legacy legacy de la app original
     * En el schema legacy no hay estado_pago, lo simulamos con estado 'inactivo'
     */
    public function scopePendientes($query)
    {
        return $query->where('estado', 'inactivo');
    }

    /**
     * Scope 'pagados' — Legacy de la app original
     * En el schema legacy no hay estado_pago, lo simulamos con estado 'activo'
     */
    public function scopePagados($query)
    {
        return $query->where('estado', 'activo');
    }

    /** Buscar por nombre, apellido o documento */
    public function scopeBuscar($query, $termino)
    {
        return $query->where(function ($q) use ($termino) {
            $q->where('nombre', 'like', "%{$termino}%")
              ->orWhere('apellido', 'like', "%{$termino}%")
              ->orWhere('numero_documento', 'like', "%{$termino}%")
              ->orWhere('telefono', 'like', "%{$termino}%");
        });
    }

    // =========================================================
    // ACCESSORS
    // =========================================================
    public function getNombreCompletoAttribute()
    {
        return "{$this->nombre} {$this->apellido}";
    }

    // =========================================================
    // RELACIONES
    // =========================================================
    public function historiasClinicas()
    {
        return $this->hasMany(HistoriaClinica::class, 'paciente_id');
    }

    public function citas()
    {
        return $this->hasMany(Cita::class, 'paciente_id');
    }
}
