<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PacienteSesion extends Model
{
    use HasFactory;

    protected $table = 'pacientes_sesiones';

    public $timestamps = false;

    protected $fillable = [
        'paciente_usuario_id',
        'token',
        'ip_address',
        'user_agent',
        'expira_en',
    ];

    protected $casts = [
        'expira_en' => 'datetime',
        'fecha_creacion' => 'datetime',
    ];

    protected $dates = [
        'expira_en',
        'fecha_creacion',
    ];

    /**
     * Relación con el usuario del paciente
     */
    public function pacienteUsuario()
    {
        return $this->belongsTo(PacienteUsuario::class, 'paciente_usuario_id');
    }

    /**
     * Scopes
     */
    public function scopeActivas($query)
    {
        return $query->where('expira_en', '>', now());
    }

    public function scopeExpiradas($query)
    {
        return $query->where('expira_en', '<=', now());
    }

    public function scopePorToken($query, $token)
    {
        return $query->where('token', $token);
    }

    /**
     * Verificar si la sesión está activa
     */
    public function estaActiva(): bool
    {
        return $this->expira_en && $this->expira_en->isFuture();
    }

    /**
     * Extender la sesión
     */
    public function extender($horas = 24): void
    {
        $this->expira_en = now()->addHours($horas);
        $this->save();
    }

    /**
     * Cerrar sesión
     */
    public function cerrar(): void
    {
        $this->delete();
    }

    /**
     * Limpiar sesiones expiradas (método estático)
     */
    public static function limpiarExpiradas(): int
    {
        return static::expiradas()->delete();
    }
}
