<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;

class PacienteUsuario extends Model
{
    use HasFactory;

    protected $table = 'pacientes_usuario';

    protected $fillable = [
        'paciente_id',
        'email',
        'password',
        'activo',
        'email_verificado',
        'token_verificacion',
        'ultimo_acceso',
        'intentos_fallidos',
        'bloqueado_hasta',
    ];

    protected $hidden = [
        'password',
        'token_verificacion',
    ];

    protected $casts = [
        'activo' => 'boolean',
        'email_verificado' => 'boolean',
        'intentos_fallidos' => 'integer',
        'ultimo_acceso' => 'datetime',
        'bloqueado_hasta' => 'datetime',
        'fecha_creacion' => 'datetime',
        'fecha_actualizacion' => 'datetime',
    ];

    protected $dates = [
        'ultimo_acceso',
        'bloqueado_hasta',
        'fecha_creacion',
        'fecha_actualizacion',
    ];

    /**
     * Relación con el paciente
     */
    public function paciente()
    {
        return $this->belongsTo(Paciente::class, 'paciente_id');
    }

    /**
     * Relación con las sesiones
     */
    public function sesiones()
    {
        return $this->hasMany(PacienteSesion::class, 'paciente_usuario_id');
    }

    /**
     * Mutator para hashear la contraseña automáticamente
     */
    public function setPasswordAttribute($value)
    {
        $this->attributes['password'] = Hash::make($value);
    }

    /**
     * Scopes
     */
    public function scopeActivos($query)
    {
        return $query->where('activo', true);
    }

    public function scopeVerificados($query)
    {
        return $query->where('email_verificado', true);
    }

    public function scopeBloqueados($query)
    {
        return $query->where('bloqueado_hasta', '>', now());
    }

    /**
     * Verificar si el usuario está bloqueado
     */
    public function estaBloqueado(): bool
    {
        return $this->bloqueado_hasta && $this->bloqueado_hasta->isFuture();
    }

    /**
     * Verificar si el usuario está activo
     */
    public function estaActivo(): bool
    {
        return $this->activo && !$this->estaBloqueado();
    }

    /**
     * Verificar contraseña
     */
    public function verificarPassword($password): bool
    {
        return Hash::check($password, $this->password);
    }

    /**
     * Registrar intento fallido
     */
    public function registrarIntentoFallido(): void
    {
        $this->intentos_fallidos++;
        
        // Bloquear después de 5 intentos fallidos
        if ($this->intentos_fallidos >= 5) {
            $this->bloqueado_hasta = now()->addMinutes(15);
        }
        
        $this->save();
    }

    /**
     * Resetear intentos fallidos
     */
    public function resetearIntentosFallidos(): void
    {
        $this->intentos_fallidos = 0;
        $this->bloqueado_hasta = null;
        $this->save();
    }

    /**
     * Registrar acceso
     */
    public function registrarAcceso(): void
    {
        $this->ultimo_acceso = now();
        $this->resetearIntentosFallidos();
        $this->save();
    }

    /**
     * Generar token de sesión
     */
    public function generarTokenSesion($ipAddress = null, $userAgent = null): string
    {
        $token = bin2hex(random_bytes(32));
        
        PacienteSesion::create([
            'paciente_usuario_id' => $this->id,
            'token' => $token,
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'expira_en' => now()->addHours(24),
        ]);
        
        return $token;
    }

    /**
     * Cerrar todas las sesiones
     */
    public function cerrarTodasLasSesiones(): void
    {
        $this->sesiones()->delete();
    }
}
