<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PagoExamen extends Model
{
    use HasFactory;

    protected $table = 'pagos_examenes';

    protected $fillable = [
        'examen_id',
        'paciente_id',
        'monto',
        'metodo_pago',
        'estado_pago',
        'fecha_pago',
        'numero_comprobante',
        'archivo_comprobante',
        'procesado_por',
        'observaciones',
    ];

    protected $casts = [
        'monto' => 'decimal:2',
        'fecha_pago' => 'datetime',
        'fecha_creacion' => 'datetime',
        'fecha_actualizacion' => 'datetime',
    ];

    protected $dates = [
        'fecha_pago',
        'fecha_creacion',
        'fecha_actualizacion',
    ];

    // Constantes para estados de pago
    const ESTADO_PENDIENTE = 'pendiente';
    const ESTADO_PAGADO = 'pagado';
    const ESTADO_PARCIAL = 'parcial';
    const ESTADO_CANCELADO = 'cancelado';
    const ESTADO_REEMBOLSADO = 'reembolsado';

    // Constantes para métodos de pago
    const METODO_EFECTIVO = 'efectivo';
    const METODO_TARJETA = 'tarjeta';
    const METODO_TRANSFERENCIA = 'transferencia';
    const METODO_SEGURO = 'seguro';
    const METODO_OTRO = 'otro';

    /**
     * Relación con el examen médico
     */
    public function examen()
    {
        return $this->belongsTo(ExamenMedico::class, 'examen_id');
    }

    /**
     * Relación con el paciente
     */
    public function paciente()
    {
        return $this->belongsTo(Paciente::class, 'paciente_id');
    }

    /**
     * Relación con el usuario que procesó el pago
     */
    public function procesador()
    {
        return $this->belongsTo(Usuario::class, 'procesado_por');
    }

    /**
     * Scopes
     */
    public function scopePagados($query)
    {
        return $query->where('estado_pago', self::ESTADO_PAGADO);
    }

    public function scopePendientes($query)
    {
        return $query->where('estado_pago', self::ESTADO_PENDIENTE);
    }

    public function scopePorPaciente($query, $pacienteId)
    {
        return $query->where('paciente_id', $pacienteId);
    }

    /**
     * Verificar si el pago está completado
     */
    public function isPagado(): bool
    {
        return $this->estado_pago === self::ESTADO_PAGADO;
    }

    /**
     * Verificar si el pago está pendiente
     */
    public function isPendiente(): bool
    {
        return $this->estado_pago === self::ESTADO_PENDIENTE;
    }

    /**
     * Marcar como pagado
     */
    public function marcarComoPagado($procesadoPor = null): bool
    {
        $this->estado_pago = self::ESTADO_PAGADO;
        $this->fecha_pago = now();
        
        if ($procesadoPor) {
            $this->procesado_por = $procesadoPor;
        }
        
        return $this->save();
    }

    /**
     * Obtener el nombre del método de pago formateado
     */
    public function getMetodoPagoFormateadoAttribute(): string
    {
        $metodos = [
            self::METODO_EFECTIVO => 'Efectivo',
            self::METODO_TARJETA => 'Tarjeta',
            self::METODO_TRANSFERENCIA => 'Transferencia',
            self::METODO_SEGURO => 'Seguro',
            self::METODO_OTRO => 'Otro',
        ];

        return $metodos[$this->metodo_pago] ?? $this->metodo_pago;
    }

    /**
     * Obtener el estado formateado
     */
    public function getEstadoFormateadoAttribute(): string
    {
        $estados = [
            self::ESTADO_PENDIENTE => 'Pendiente',
            self::ESTADO_PAGADO => 'Pagado',
            self::ESTADO_PARCIAL => 'Pago Parcial',
            self::ESTADO_CANCELADO => 'Cancelado',
            self::ESTADO_REEMBOLSADO => 'Reembolsado',
        ];

        return $estados[$this->estado_pago] ?? $this->estado_pago;
    }
}
