<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class Usuario extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * La tabla asociada con el modelo.
     */
    protected $table = 'usuarios';

    /**
     * Los atributos que son asignables en masa.
     */
    protected $fillable = [
        'nombre',
        'apellido',
        'email',
        'password',
        'telefono',
        'horario',
        'rol',
        'especialidad',
        'numero_licencia',
        'foto_perfil',
        'estado',
    ];

    /**
     * Los atributos que deben estar ocultos para la serialización.
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Los atributos que se agregan al modelo.
     */
    protected $appends = ['nombre_completo'];

    /**
     * Los atributos que deben ser cast.
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'fecha_creacion' => 'datetime',
        'fecha_actualizacion' => 'datetime',
        'ultimo_acceso' => 'datetime',
    ];

    /**
     * Nombre de las columnas de timestamps personalizadas
     */
    const CREATED_AT = 'fecha_creacion';
    const UPDATED_AT = 'fecha_actualizacion';

    /**
     * Relaciones
     */

    // Pacientes registrados por este usuario
    public function pacientesRegistrados()
    {
        return $this->hasMany(Paciente::class, 'registrado_por');
    }

    // Citas como médico
    public function citasComoMedico()
    {
        return $this->hasMany(Cita::class, 'medico_id');
    }

    // Citas creadas
    public function citasCreadas()
    {
        return $this->hasMany(Cita::class, 'creada_por');
    }

    // Historias clínicas como médico
    public function historiasClinicas()
    {
        return $this->hasMany(HistoriaClinica::class, 'medico_id');
    }

    // Vacunas aplicadas
    public function vacunasAplicadas()
    {
        return $this->hasMany(Vacuna::class, 'profesional_id');
    }

    // Exámenes solicitados
    public function examenesSolicitados()
    {
        return $this->hasMany(ExamenMedico::class, 'medico_solicitante_id');
    }

    // Recetas emitidas
    public function recetasEmitidas()
    {
        return $this->hasMany(RecetaMedica::class, 'medico_id');
    }

    // Facturas generadas
    public function facturasGeneradas()
    {
        return $this->hasMany(Facturacion::class, 'generada_por');
    }

    // Acciones de auditoría
    public function accionesAuditoria()
    {
        return $this->hasMany(Auditoria::class, 'usuario_id');
    }

    /**
     * Accesor para nombre completo
     */
    public function getNombreCompletoAttribute()
    {
        return "{$this->nombre} {$this->apellido}";
    }

    /**
     * Scopes
     */
    public function scopeActivos($query)
    {
        return $query->where('estado', 'activo');
    }

    public function scopeMedicos($query)
    {
        return $query->where('rol', 'medico');
    }

    public function scopeEnfermeros($query)
    {
        return $query->where('rol', 'enfermero');
    }

    public function scopeAdministradores($query)
    {
        return $query->where('rol', 'admin');
    }

    /**
     * Verificar si el usuario es médico
     */
    public function esMedico()
    {
        return $this->rol === 'medico';
    }

    /**
     * Verificar si el usuario es admin
     */
    public function esAdmin()
    {
        return $this->rol === 'admin';
    }

    /**
     * Actualizar último acceso
     */
    public function actualizarUltimoAcceso()
    {
        $this->ultimo_acceso = now();
        $this->save();
    }
}
