-- =============================================
-- EJEMPLOS DE CONSULTAS SQL
-- Cibertronia Salud - Base de Datos
-- =============================================

-- =============================================
-- CONSULTAS BÁSICAS
-- =============================================

-- Ver todos los pacientes activos
SELECT 
    numero_historia,
    CONCAT(nombre, ' ', apellido) AS nombre_completo,
    edad,
    sexo,
    telefono,
    grupo_sanguineo
FROM pacientes
WHERE estado = 'activo'
ORDER BY apellido, nombre;

-- Ver todos los médicos
SELECT 
    id,
    CONCAT(nombre, ' ', apellido) AS nombre_completo,
    especialidad,
    email,
    telefono
FROM usuarios
WHERE rol = 'medico' AND estado = 'activo'
ORDER BY nombre;

-- Ver citas de hoy
SELECT 
    c.fecha_cita,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.numero_historia,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    u.especialidad,
    c.tipo_cita,
    c.estado,
    c.sala_consultorio
FROM citas c
INNER JOIN pacientes p ON c.paciente_id = p.id
INNER JOIN usuarios u ON c.medico_id = u.id
WHERE DATE(c.fecha_cita) = CURDATE()
ORDER BY c.fecha_cita;

-- =============================================
-- CONSULTAS DE HISTORIAL MÉDICO
-- =============================================

-- Ver historial completo de un paciente (ID = 1)
SELECT 
    hc.fecha_consulta,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    u.especialidad,
    hc.motivo_consulta,
    hc.diagnostico,
    hc.tratamiento,
    hc.presion_arterial,
    hc.temperatura,
    hc.peso,
    hc.imc
FROM historias_clinicas hc
INNER JOIN usuarios u ON hc.medico_id = u.id
WHERE hc.paciente_id = 1
ORDER BY hc.fecha_consulta DESC;

-- Ver antecedentes médicos de un paciente
SELECT 
    tipo,
    descripcion,
    fecha_diagnostico,
    tratamiento_actual,
    observaciones
FROM antecedentes_medicos
WHERE paciente_id = 1
ORDER BY fecha_diagnostico DESC;

-- Ver vacunas de un paciente
SELECT 
    nombre_vacuna,
    dosis_numero,
    fecha_aplicacion,
    lote,
    laboratorio,
    proxima_dosis,
    CONCAT(u.nombre, ' ', u.apellido) AS aplicada_por
FROM vacunas v
LEFT JOIN usuarios u ON v.profesional_id = u.id
WHERE v.paciente_id = 1
ORDER BY fecha_aplicacion DESC;

-- =============================================
-- CONSULTAS DE RECETAS Y MEDICAMENTOS
-- =============================================

-- Ver recetas activas de un paciente
SELECT 
    r.id,
    r.fecha_emision,
    r.fecha_vencimiento,
    r.estado,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    r.instrucciones_generales
FROM recetas_medicas r
INNER JOIN usuarios u ON r.medico_id = u.id
WHERE r.paciente_id = 1 
  AND r.estado = 'activa'
  AND r.fecha_vencimiento >= CURDATE()
ORDER BY r.fecha_emision DESC;

-- Ver medicamentos de una receta específica
SELECT 
    nombre_medicamento,
    dosis,
    via_administracion,
    frecuencia,
    duracion_dias,
    cantidad_total,
    instrucciones_especiales
FROM medicamentos_receta
WHERE receta_id = 1;

-- Ver todas las recetas con sus medicamentos
SELECT 
    r.fecha_emision,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    m.nombre_medicamento,
    m.dosis,
    m.frecuencia,
    m.duracion_dias
FROM recetas_medicas r
INNER JOIN pacientes p ON r.paciente_id = p.id
INNER JOIN usuarios u ON r.medico_id = u.id
INNER JOIN medicamentos_receta m ON r.id = m.receta_id
WHERE r.estado = 'activa'
ORDER BY r.fecha_emision DESC;

-- =============================================
-- CONSULTAS DE CITAS Y PROGRAMACIÓN
-- =============================================

-- Ver agenda de un médico específico (ID = 2)
SELECT 
    c.fecha_cita,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.telefono,
    c.tipo_cita,
    c.motivo_consulta,
    c.sala_consultorio,
    c.estado
FROM citas c
INNER JOIN pacientes p ON c.paciente_id = p.id
WHERE c.medico_id = 2
  AND c.fecha_cita >= CURDATE()
  AND c.estado NOT IN ('cancelada', 'completada')
ORDER BY c.fecha_cita;

-- Ver citas pendientes de la semana
SELECT 
    DATE_FORMAT(c.fecha_cita, '%W %d/%m/%Y %H:%i') AS fecha_hora,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    c.tipo_cita,
    c.estado
FROM citas c
INNER JOIN pacientes p ON c.paciente_id = p.id
INNER JOIN usuarios u ON c.medico_id = u.id
WHERE YEARWEEK(c.fecha_cita, 1) = YEARWEEK(CURDATE(), 1)
  AND c.estado IN ('programada', 'confirmada')
ORDER BY c.fecha_cita;

-- Buscar horarios disponibles de un médico
SELECT 
    DATE(fecha_cita) AS fecha,
    TIME(fecha_cita) AS hora_inicio,
    TIME(DATE_ADD(fecha_cita, INTERVAL duracion_minutos MINUTE)) AS hora_fin
FROM citas
WHERE medico_id = 2
  AND DATE(fecha_cita) = '2024-02-15'
  AND estado NOT IN ('cancelada')
ORDER BY fecha_cita;

-- =============================================
-- CONSULTAS DE EXÁMENES MÉDICOS
-- =============================================

-- Ver exámenes pendientes
SELECT 
    e.fecha_solicitud,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    e.tipo_examen,
    e.descripcion,
    CONCAT(u.nombre, ' ', u.apellido) AS medico_solicitante,
    e.estado
FROM examenes_medicos e
INNER JOIN pacientes p ON e.paciente_id = p.id
INNER JOIN usuarios u ON e.medico_solicitante_id = u.id
WHERE e.estado IN ('solicitado', 'en_proceso')
ORDER BY e.fecha_solicitud;

-- Ver resultados de exámenes de un paciente
SELECT 
    e.fecha_realizacion,
    e.tipo_examen,
    e.laboratorio_centro,
    e.resultados,
    e.archivo_resultado,
    CONCAT(u.nombre, ' ', u.apellido) AS solicitado_por
FROM examenes_medicos e
INNER JOIN usuarios u ON e.medico_solicitante_id = u.id
WHERE e.paciente_id = 1
  AND e.estado = 'completado'
ORDER BY e.fecha_realizacion DESC;

-- =============================================
-- CONSULTAS DE FACTURACIÓN
-- =============================================

-- Ver facturas pendientes de pago
SELECT 
    f.numero_factura,
    f.fecha_emision,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    f.concepto,
    f.total,
    f.estado,
    DATEDIFF(CURDATE(), f.fecha_emision) AS dias_pendiente
FROM facturacion f
INNER JOIN pacientes p ON f.paciente_id = p.id
WHERE f.estado IN ('pendiente', 'parcial')
ORDER BY f.fecha_emision;

-- Ver total facturado en el mes actual
SELECT 
    SUM(total) AS total_mes,
    COUNT(*) AS cantidad_facturas,
    AVG(total) AS promedio_factura
FROM facturacion
WHERE YEAR(fecha_emision) = YEAR(CURDATE())
  AND MONTH(fecha_emision) = MONTH(CURDATE())
  AND estado != 'cancelada';

-- Ver facturas por método de pago
SELECT 
    metodo_pago,
    COUNT(*) AS cantidad,
    SUM(total) AS total
FROM facturacion
WHERE estado = 'pagada'
  AND YEAR(fecha_emision) = YEAR(CURDATE())
  AND MONTH(fecha_emision) = MONTH(CURDATE())
GROUP BY metodo_pago
ORDER BY total DESC;

-- =============================================
-- CONSULTAS ESTADÍSTICAS
-- =============================================

-- Pacientes por grupo sanguíneo
SELECT 
    grupo_sanguineo,
    COUNT(*) AS cantidad,
    ROUND(COUNT(*) * 100.0 / (SELECT COUNT(*) FROM pacientes WHERE estado = 'activo'), 2) AS porcentaje
FROM pacientes
WHERE estado = 'activo'
  AND grupo_sanguineo IS NOT NULL
GROUP BY grupo_sanguineo
ORDER BY cantidad DESC;

-- Citas por estado del mes actual
SELECT 
    estado,
    COUNT(*) AS cantidad
FROM citas
WHERE YEAR(fecha_cita) = YEAR(CURDATE())
  AND MONTH(fecha_cita) = MONTH(CURDATE())
GROUP BY estado
ORDER BY cantidad DESC;

-- Top 5 médicos con más consultas
SELECT 
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    u.especialidad,
    COUNT(c.id) AS total_consultas
FROM usuarios u
INNER JOIN citas c ON u.id = c.medico_id
WHERE u.rol = 'medico'
  AND c.estado = 'completada'
  AND YEAR(c.fecha_cita) = YEAR(CURDATE())
GROUP BY u.id, u.nombre, u.apellido, u.especialidad
ORDER BY total_consultas DESC
LIMIT 5;

-- Pacientes con más visitas
SELECT 
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.numero_historia,
    COUNT(c.id) AS total_visitas,
    MAX(c.fecha_cita) AS ultima_visita
FROM pacientes p
LEFT JOIN citas c ON p.id = c.paciente_id
WHERE p.estado = 'activo'
GROUP BY p.id, p.nombre, p.apellido, p.numero_historia
HAVING total_visitas > 0
ORDER BY total_visitas DESC
LIMIT 10;

-- =============================================
-- BÚSQUEDAS AVANZADAS
-- =============================================

-- Buscar paciente por nombre o número de documento
SELECT 
    numero_historia,
    CONCAT(nombre, ' ', apellido) AS nombre_completo,
    numero_documento,
    telefono,
    email
FROM pacientes
WHERE (CONCAT(nombre, ' ', apellido) LIKE '%Carlos%'
   OR numero_documento LIKE '%12345%')
  AND estado = 'activo';

-- Buscar pacientes con alergias específicas
SELECT 
    numero_historia,
    CONCAT(nombre, ' ', apellido) AS nombre_completo,
    alergias,
    telefono
FROM pacientes
WHERE alergias IS NOT NULL
  AND alergias LIKE '%penicilina%'
  AND estado = 'activo';

-- Buscar historias clínicas por diagnóstico
SELECT 
    hc.fecha_consulta,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.numero_historia,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    hc.diagnostico
FROM historias_clinicas hc
INNER JOIN pacientes p ON hc.paciente_id = p.id
INNER JOIN usuarios u ON hc.medico_id = u.id
WHERE hc.diagnostico LIKE '%diabetes%'
ORDER BY hc.fecha_consulta DESC;

-- =============================================
-- ALERTAS Y RECORDATORIOS
-- =============================================

-- Pacientes con citas para mañana (para enviar recordatorios)
SELECT 
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.telefono,
    p.email,
    c.fecha_cita,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    c.sala_consultorio
FROM citas c
INNER JOIN pacientes p ON c.paciente_id = p.id
INNER JOIN usuarios u ON c.medico_id = u.id
WHERE DATE(c.fecha_cita) = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
  AND c.estado IN ('programada', 'confirmada')
ORDER BY c.fecha_cita;

-- Pacientes que no han tenido cita en los últimos 6 meses
SELECT 
    p.numero_historia,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.telefono,
    p.email,
    MAX(c.fecha_cita) AS ultima_cita,
    DATEDIFF(CURDATE(), MAX(c.fecha_cita)) AS dias_sin_cita
FROM pacientes p
LEFT JOIN citas c ON p.id = c.paciente_id
WHERE p.estado = 'activo'
GROUP BY p.id, p.numero_historia, p.nombre, p.apellido, p.telefono, p.email
HAVING ultima_cita IS NULL 
   OR ultima_cita < DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
ORDER BY ultima_cita DESC;

-- Recetas próximas a vencer (en los próximos 5 días)
SELECT 
    r.fecha_vencimiento,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.telefono,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    r.instrucciones_generales
FROM recetas_medicas r
INNER JOIN pacientes p ON r.paciente_id = p.id
INNER JOIN usuarios u ON r.medico_id = u.id
WHERE r.estado = 'activa'
  AND r.fecha_vencimiento BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 5 DAY)
ORDER BY r.fecha_vencimiento;

-- =============================================
-- AUDITORÍA
-- =============================================

-- Ver últimas acciones en el sistema
SELECT 
    a.fecha_accion,
    CONCAT(u.nombre, ' ', u.apellido) AS usuario,
    u.rol,
    a.accion,
    a.tabla_afectada,
    a.descripcion
FROM auditoria a
LEFT JOIN usuarios u ON a.usuario_id = u.id
ORDER BY a.fecha_accion DESC
LIMIT 50;

-- Ver acciones de un usuario específico
SELECT 
    fecha_accion,
    accion,
    tabla_afectada,
    registro_id,
    descripcion
FROM auditoria
WHERE usuario_id = 1
ORDER BY fecha_accion DESC
LIMIT 20;

-- =============================================
-- USO DE VISTAS PREDEFINIDAS
-- =============================================

-- Ver todas las citas de hoy usando la vista
SELECT * FROM vista_citas_hoy;

-- Ver pacientes con su última cita
SELECT * FROM vista_pacientes_ultima_cita 
WHERE estado = 'activo'
ORDER BY ultima_cita DESC;

-- Ver resumen de historias clínicas
SELECT * FROM vista_historias_clinicas_resumen
LIMIT 20;

-- =============================================
-- USO DE PROCEDIMIENTOS ALMACENADOS
-- =============================================

-- Obtener historial completo de un paciente
CALL obtener_historial_paciente(1);

-- Generar nuevo número de historia
CALL generar_numero_historia();

-- Agendar una nueva cita
CALL agendar_cita(
    1,                              -- paciente_id
    2,                              -- medico_id
    '2024-02-15 10:00:00',         -- fecha_cita
    30,                             -- duracion_minutos
    'consulta_general',             -- tipo_cita
    'Control de rutina',            -- motivo_consulta
    'Consultorio 1',                -- sala_consultorio
    1                               -- creada_por (usuario_id)
);

-- =============================================
-- FIN DE EJEMPLOS
-- =============================================
