-- =============================================
-- BASE DE DATOS: CIBERTRONIA SALUD
-- Sistema de Registro de Pacientes
-- =============================================

-- Crear base de datos
DROP DATABASE IF EXISTS cibertronia_salud;
CREATE DATABASE cibertronia_salud CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE cibertronia_salud;

-- =============================================
-- TABLA: usuarios
-- Gestión de usuarios del sistema (médicos, enfermeros, administradores)
-- =============================================
CREATE TABLE usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nombre VARCHAR(100) NOT NULL,
    apellido VARCHAR(100) NOT NULL,
    email VARCHAR(150) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    telefono VARCHAR(20),
    rol ENUM('admin', 'medico', 'enfermero', 'recepcionista') DEFAULT 'recepcionista',
    especialidad VARCHAR(100) NULL,
    numero_licencia VARCHAR(50) NULL,
    foto_perfil VARCHAR(255) NULL,
    estado ENUM('activo', 'inactivo', 'suspendido') DEFAULT 'activo',
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    ultimo_acceso TIMESTAMP NULL,
    INDEX idx_email (email),
    INDEX idx_rol (rol),
    INDEX idx_estado (estado)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: pacientes
-- Información general de pacientes
-- =============================================
CREATE TABLE pacientes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    numero_historia VARCHAR(20) UNIQUE NOT NULL,
    nombre VARCHAR(100) NOT NULL,
    apellido VARCHAR(100) NOT NULL,
    fecha_nacimiento DATE NOT NULL,
    edad INT GENERATED ALWAYS AS (YEAR(CURDATE()) - YEAR(fecha_nacimiento)) VIRTUAL,
    sexo ENUM('M', 'F', 'Otro') NOT NULL,
    tipo_documento ENUM('DNI', 'Pasaporte', 'Carnet de Extranjería', 'Otro') DEFAULT 'DNI',
    numero_documento VARCHAR(50) UNIQUE NOT NULL,
    telefono VARCHAR(20),
    telefono_emergencia VARCHAR(20),
    contacto_emergencia VARCHAR(150),
    email VARCHAR(150),
    direccion TEXT,
    ciudad VARCHAR(100),
    codigo_postal VARCHAR(10),
    ocupacion VARCHAR(100),
    estado_civil ENUM('Soltero', 'Casado', 'Divorciado', 'Viudo', 'Unión Libre') NULL,
    grupo_sanguineo ENUM('A+', 'A-', 'B+', 'B-', 'AB+', 'AB-', 'O+', 'O-') NULL,
    alergias TEXT NULL,
    observaciones TEXT NULL,
    foto_paciente VARCHAR(255) NULL,
    estado ENUM('activo', 'inactivo', 'fallecido') DEFAULT 'activo',
    fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    registrado_por INT NULL,
    INDEX idx_numero_historia (numero_historia),
    INDEX idx_numero_documento (numero_documento),
    INDEX idx_nombre_apellido (nombre, apellido),
    INDEX idx_estado (estado),
    FOREIGN KEY (registrado_por) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: citas
-- Gestión de citas médicas
-- =============================================
CREATE TABLE citas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    paciente_id INT NOT NULL,
    medico_id INT NOT NULL,
    fecha_cita DATETIME NOT NULL,
    duracion_minutos INT DEFAULT 30,
    tipo_cita ENUM('consulta_general', 'consulta_especializada', 'seguimiento', 'emergencia', 'control') DEFAULT 'consulta_general',
    motivo_consulta TEXT NOT NULL,
    estado ENUM('programada', 'confirmada', 'en_curso', 'completada', 'cancelada', 'no_asistio') DEFAULT 'programada',
    sala_consultorio VARCHAR(50) NULL,
    observaciones TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    creada_por INT NULL,
    INDEX idx_paciente (paciente_id),
    INDEX idx_medico (medico_id),
    INDEX idx_fecha_cita (fecha_cita),
    INDEX idx_estado (estado),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (medico_id) REFERENCES usuarios(id) ON DELETE RESTRICT,
    FOREIGN KEY (creada_por) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: historias_clinicas
-- Registro de historias clínicas por consulta
-- =============================================
CREATE TABLE historias_clinicas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    cita_id INT NOT NULL UNIQUE,
    paciente_id INT NOT NULL,
    medico_id INT NOT NULL,
    fecha_consulta DATETIME NOT NULL,
    
    -- Signos Vitales
    presion_arterial VARCHAR(20) NULL COMMENT 'Ej: 120/80',
    frecuencia_cardiaca INT NULL COMMENT 'pulsaciones por minuto',
    temperatura DECIMAL(4,2) NULL COMMENT 'grados celsius',
    frecuencia_respiratoria INT NULL COMMENT 'respiraciones por minuto',
    peso DECIMAL(5,2) NULL COMMENT 'kilogramos',
    altura DECIMAL(5,2) NULL COMMENT 'centímetros',
    imc DECIMAL(5,2) GENERATED ALWAYS AS (peso / POWER(altura/100, 2)) VIRTUAL,
    saturacion_oxigeno INT NULL COMMENT 'porcentaje',
    
    -- Consulta
    motivo_consulta TEXT NOT NULL,
    sintomas TEXT NULL,
    diagnostico TEXT NOT NULL,
    tratamiento TEXT NULL,
    observaciones TEXT NULL,
    examenes_solicitados TEXT NULL,
    medicamentos_recetados TEXT NULL,
    
    -- Seguimiento
    proxima_cita DATE NULL,
    recomendaciones TEXT NULL,
    
    -- Archivos adjuntos
    archivos_adjuntos JSON NULL COMMENT 'Array de rutas de archivos',
    
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_cita (cita_id),
    INDEX idx_paciente (paciente_id),
    INDEX idx_medico (medico_id),
    INDEX idx_fecha_consulta (fecha_consulta),
    FOREIGN KEY (cita_id) REFERENCES citas(id) ON DELETE CASCADE,
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (medico_id) REFERENCES usuarios(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: recetas_medicas
-- Gestión de recetas médicas
-- =============================================
CREATE TABLE recetas_medicas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    historia_clinica_id INT NOT NULL,
    paciente_id INT NOT NULL,
    medico_id INT NOT NULL,
    fecha_emision DATE NOT NULL,
    vigencia_dias INT DEFAULT 30,
    fecha_vencimiento DATE GENERATED ALWAYS AS (DATE_ADD(fecha_emision, INTERVAL vigencia_dias DAY)) VIRTUAL,
    instrucciones_generales TEXT NULL,
    estado ENUM('activa', 'utilizada', 'vencida', 'cancelada') DEFAULT 'activa',
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_historia_clinica (historia_clinica_id),
    INDEX idx_paciente (paciente_id),
    INDEX idx_medico (medico_id),
    INDEX idx_estado (estado),
    FOREIGN KEY (historia_clinica_id) REFERENCES historias_clinicas(id) ON DELETE CASCADE,
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (medico_id) REFERENCES usuarios(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: medicamentos_receta
-- Detalle de medicamentos en cada receta
-- =============================================
CREATE TABLE medicamentos_receta (
    id INT AUTO_INCREMENT PRIMARY KEY,
    receta_id INT NOT NULL,
    nombre_medicamento VARCHAR(200) NOT NULL,
    dosis VARCHAR(100) NOT NULL,
    via_administracion ENUM('oral', 'intravenosa', 'intramuscular', 'subcutánea', 'tópica', 'inhalatoria', 'oftálmica', 'ótica', 'rectal', 'otra') DEFAULT 'oral',
    frecuencia VARCHAR(100) NOT NULL COMMENT 'Ej: cada 8 horas, 3 veces al día',
    duracion_dias INT NOT NULL,
    cantidad_total INT NOT NULL,
    instrucciones_especiales TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_receta (receta_id),
    FOREIGN KEY (receta_id) REFERENCES recetas_medicas(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: examenes_medicos
-- Registro de exámenes y estudios médicos
-- =============================================
CREATE TABLE examenes_medicos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    paciente_id INT NOT NULL,
    medico_solicitante_id INT NOT NULL,
    historia_clinica_id INT NULL,
    tipo_examen VARCHAR(150) NOT NULL,
    descripcion TEXT NULL,
    fecha_solicitud DATE NOT NULL,
    fecha_realizacion DATE NULL,
    laboratorio_centro VARCHAR(200) NULL,
    resultados TEXT NULL,
    archivo_resultado VARCHAR(255) NULL,
    estado ENUM('solicitado', 'en_proceso', 'completado', 'cancelado') DEFAULT 'solicitado',
    observaciones TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_paciente (paciente_id),
    INDEX idx_medico_solicitante (medico_solicitante_id),
    INDEX idx_historia_clinica (historia_clinica_id),
    INDEX idx_estado (estado),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (medico_solicitante_id) REFERENCES usuarios(id) ON DELETE RESTRICT,
    FOREIGN KEY (historia_clinica_id) REFERENCES historias_clinicas(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: vacunas
-- Registro de vacunación de pacientes
-- =============================================
CREATE TABLE vacunas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    paciente_id INT NOT NULL,
    nombre_vacuna VARCHAR(150) NOT NULL,
    dosis_numero INT NOT NULL COMMENT 'número de dosis (1ra, 2da, etc)',
    fecha_aplicacion DATE NOT NULL,
    lote VARCHAR(50) NULL,
    laboratorio VARCHAR(100) NULL,
    profesional_id INT NULL,
    centro_vacunacion VARCHAR(200) NULL,
    proxima_dosis DATE NULL,
    reacciones_adversas TEXT NULL,
    observaciones TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_paciente (paciente_id),
    INDEX idx_profesional (profesional_id),
    INDEX idx_fecha_aplicacion (fecha_aplicacion),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (profesional_id) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: antecedentes_medicos
-- Antecedentes médicos del paciente
-- =============================================
CREATE TABLE antecedentes_medicos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    paciente_id INT NOT NULL,
    tipo ENUM('personal', 'familiar', 'quirurgico', 'hospitalario', 'traumatico', 'gineco-obstetrico', 'otro') NOT NULL,
    descripcion TEXT NOT NULL,
    fecha_diagnostico DATE NULL,
    tratamiento_actual BOOLEAN DEFAULT FALSE,
    observaciones TEXT NULL,
    fecha_registro TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    registrado_por INT NULL,
    INDEX idx_paciente (paciente_id),
    INDEX idx_tipo (tipo),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE,
    FOREIGN KEY (registrado_por) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: seguros_medicos
-- Información de seguros médicos de pacientes
-- =============================================
CREATE TABLE seguros_medicos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    paciente_id INT NOT NULL,
    compania_seguro VARCHAR(150) NOT NULL,
    numero_poliza VARCHAR(100) NOT NULL,
    tipo_plan VARCHAR(100) NULL,
    vigencia_desde DATE NOT NULL,
    vigencia_hasta DATE NOT NULL,
    estado ENUM('activo', 'vencido', 'cancelado') DEFAULT 'activo',
    cobertura_porcentaje INT NULL,
    observaciones TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_paciente (paciente_id),
    INDEX idx_estado (estado),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: facturacion
-- Gestión de facturación y pagos
-- =============================================
CREATE TABLE facturacion (
    id INT AUTO_INCREMENT PRIMARY KEY,
    numero_factura VARCHAR(50) UNIQUE NOT NULL,
    paciente_id INT NOT NULL,
    cita_id INT NULL,
    fecha_emision DATE NOT NULL,
    concepto TEXT NOT NULL,
    subtotal DECIMAL(10,2) NOT NULL,
    impuesto DECIMAL(10,2) DEFAULT 0.00,
    descuento DECIMAL(10,2) DEFAULT 0.00,
    total DECIMAL(10,2) NOT NULL,
    estado ENUM('pendiente', 'pagada', 'parcial', 'cancelada', 'vencida') DEFAULT 'pendiente',
    metodo_pago ENUM('efectivo', 'tarjeta', 'transferencia', 'cheque', 'seguro', 'otro') NULL,
    fecha_pago DATETIME NULL,
    observaciones TEXT NULL,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_actualizacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    generada_por INT NULL,
    INDEX idx_numero_factura (numero_factura),
    INDEX idx_paciente (paciente_id),
    INDEX idx_cita (cita_id),
    INDEX idx_estado (estado),
    INDEX idx_fecha_emision (fecha_emision),
    FOREIGN KEY (paciente_id) REFERENCES pacientes(id) ON DELETE RESTRICT,
    FOREIGN KEY (cita_id) REFERENCES citas(id) ON DELETE SET NULL,
    FOREIGN KEY (generada_por) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABLA: auditoria
-- Registro de auditoría del sistema
-- =============================================
CREATE TABLE auditoria (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NULL,
    accion VARCHAR(100) NOT NULL,
    tabla_afectada VARCHAR(100) NOT NULL,
    registro_id INT NOT NULL,
    descripcion TEXT NULL,
    datos_anteriores JSON NULL,
    datos_nuevos JSON NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    fecha_accion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_usuario (usuario_id),
    INDEX idx_tabla_afectada (tabla_afectada),
    INDEX idx_fecha_accion (fecha_accion),
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- DATOS DE EJEMPLO
-- =============================================

-- Insertar usuario administrador (password: Admin123!)
INSERT INTO usuarios (nombre, apellido, email, password, telefono, rol, estado) VALUES
('Admin', 'Sistema', 'admin@cibertroniasalud.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '999888777', 'admin', 'activo'),
('Dr. Juan', 'Pérez García', 'dr.perez@cibertroniasalud.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '999111222', 'medico', 'activo'),
('Dra. María', 'González López', 'dra.gonzalez@cibertroniasalud.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '999333444', 'medico', 'activo'),
('Carmen', 'Rodríguez Silva', 'enfermera@cibertroniasalud.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '999555666', 'enfermero', 'activo'),
('Laura', 'Martínez Vega', 'recepcion@cibertroniasalud.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', '999777888', 'recepcionista', 'activo');

-- Actualizar especialidad de médicos
UPDATE usuarios SET especialidad = 'Medicina General' WHERE id = 2;
UPDATE usuarios SET especialidad = 'Pediatría' WHERE id = 3;

-- Insertar pacientes de ejemplo
INSERT INTO pacientes (numero_historia, nombre, apellido, fecha_nacimiento, sexo, tipo_documento, numero_documento, telefono, telefono_emergencia, contacto_emergencia, email, direccion, ciudad, grupo_sanguineo, registrado_por) VALUES
('HC-2024-001', 'Carlos', 'Fernández Torres', '1985-03-15', 'M', 'DNI', '12345678', '987654321', '987654322', 'María Fernández', 'carlos.fernandez@email.com', 'Av. Principal 123', 'Lima', 'O+', 1),
('HC-2024-002', 'Ana', 'Sánchez Ruiz', '1990-07-22', 'F', 'DNI', '23456789', '987654323', '987654324', 'Pedro Sánchez', 'ana.sanchez@email.com', 'Calle Los Olivos 456', 'Lima', 'A+', 1),
('HC-2024-003', 'Luis', 'Ramírez Castro', '1978-11-30', 'M', 'DNI', '34567890', '987654325', '987654326', 'Rosa Ramírez', 'luis.ramirez@email.com', 'Jr. Las Flores 789', 'Lima', 'B+', 1);

-- =============================================
-- VISTAS ÚTILES
-- =============================================

-- Vista de pacientes con su última cita
CREATE VIEW vista_pacientes_ultima_cita AS
SELECT 
    p.id,
    p.numero_historia,
    CONCAT(p.nombre, ' ', p.apellido) AS nombre_completo,
    p.edad,
    p.sexo,
    p.telefono,
    p.email,
    MAX(c.fecha_cita) AS ultima_cita,
    p.estado
FROM pacientes p
LEFT JOIN citas c ON p.id = c.paciente_id
GROUP BY p.id, p.numero_historia, p.nombre, p.apellido, p.edad, p.sexo, p.telefono, p.email, p.estado;

-- Vista de citas del día
CREATE VIEW vista_citas_hoy AS
SELECT 
    c.id,
    c.fecha_cita,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.numero_historia,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    u.especialidad,
    c.tipo_cita,
    c.motivo_consulta,
    c.sala_consultorio,
    c.estado
FROM citas c
INNER JOIN pacientes p ON c.paciente_id = p.id
INNER JOIN usuarios u ON c.medico_id = u.id
WHERE DATE(c.fecha_cita) = CURDATE()
ORDER BY c.fecha_cita;

-- Vista de resumen de historias clínicas
CREATE VIEW vista_historias_clinicas_resumen AS
SELECT 
    hc.id,
    hc.fecha_consulta,
    CONCAT(p.nombre, ' ', p.apellido) AS paciente,
    p.numero_historia,
    CONCAT(u.nombre, ' ', u.apellido) AS medico,
    hc.diagnostico,
    hc.presion_arterial,
    hc.temperatura,
    hc.peso,
    hc.imc
FROM historias_clinicas hc
INNER JOIN pacientes p ON hc.paciente_id = p.id
INNER JOIN usuarios u ON hc.medico_id = u.id
ORDER BY hc.fecha_consulta DESC;

-- =============================================
-- PROCEDIMIENTOS ALMACENADOS
-- =============================================

DELIMITER //

-- Procedimiento para obtener el historial médico completo de un paciente
CREATE PROCEDURE obtener_historial_paciente(IN paciente_id_param INT)
BEGIN
    -- Información del paciente
    SELECT * FROM pacientes WHERE id = paciente_id_param;
    
    -- Antecedentes médicos
    SELECT * FROM antecedentes_medicos WHERE paciente_id = paciente_id_param ORDER BY fecha_registro DESC;
    
    -- Vacunas
    SELECT * FROM vacunas WHERE paciente_id = paciente_id_param ORDER BY fecha_aplicacion DESC;
    
    -- Historias clínicas
    SELECT 
        hc.*,
        CONCAT(u.nombre, ' ', u.apellido) AS nombre_medico,
        u.especialidad
    FROM historias_clinicas hc
    INNER JOIN usuarios u ON hc.medico_id = u.id
    WHERE hc.paciente_id = paciente_id_param
    ORDER BY hc.fecha_consulta DESC;
    
    -- Exámenes médicos
    SELECT * FROM examenes_medicos WHERE paciente_id = paciente_id_param ORDER BY fecha_solicitud DESC;
END //

-- Procedimiento para generar número de historia clínica automático
CREATE PROCEDURE generar_numero_historia()
BEGIN
    DECLARE ultimo_numero INT;
    DECLARE nuevo_numero VARCHAR(20);
    DECLARE anio_actual VARCHAR(4);
    
    SET anio_actual = YEAR(CURDATE());
    
    SELECT COALESCE(MAX(CAST(SUBSTRING(numero_historia, 9) AS UNSIGNED)), 0) INTO ultimo_numero
    FROM pacientes
    WHERE numero_historia LIKE CONCAT('HC-', anio_actual, '-%');
    
    SET nuevo_numero = CONCAT('HC-', anio_actual, '-', LPAD(ultimo_numero + 1, 3, '0'));
    
    SELECT nuevo_numero AS numero_historia;
END //

-- Procedimiento para agendar cita con validación
CREATE PROCEDURE agendar_cita(
    IN p_paciente_id INT,
    IN p_medico_id INT,
    IN p_fecha_cita DATETIME,
    IN p_duracion_minutos INT,
    IN p_tipo_cita VARCHAR(50),
    IN p_motivo_consulta TEXT,
    IN p_sala_consultorio VARCHAR(50),
    IN p_creada_por INT
)
BEGIN
    DECLARE conflicto INT DEFAULT 0;
    
    -- Verificar conflictos de horario
    SELECT COUNT(*) INTO conflicto
    FROM citas
    WHERE medico_id = p_medico_id
    AND estado NOT IN ('cancelada', 'completada')
    AND (
        (p_fecha_cita BETWEEN fecha_cita AND DATE_ADD(fecha_cita, INTERVAL duracion_minutos MINUTE))
        OR
        (DATE_ADD(p_fecha_cita, INTERVAL p_duracion_minutos MINUTE) BETWEEN fecha_cita AND DATE_ADD(fecha_cita, INTERVAL duracion_minutos MINUTE))
    );
    
    IF conflicto > 0 THEN
        SIGNAL SQLSTATE '45000'
        SET MESSAGE_TEXT = 'El médico ya tiene una cita programada en ese horario';
    ELSE
        INSERT INTO citas (paciente_id, medico_id, fecha_cita, duracion_minutos, tipo_cita, motivo_consulta, sala_consultorio, creada_por)
        VALUES (p_paciente_id, p_medico_id, p_fecha_cita, p_duracion_minutos, p_tipo_cita, p_motivo_consulta, p_sala_consultorio, p_creada_por);
        
        SELECT LAST_INSERT_ID() AS cita_id, 'Cita agendada exitosamente' AS mensaje;
    END IF;
END //

DELIMITER ;

-- =============================================
-- FIN DEL SCRIPT
-- =============================================

-- Mostrar resumen de tablas creadas
SELECT 'Base de datos creada exitosamente' AS mensaje;
SELECT COUNT(*) AS total_tablas FROM information_schema.tables WHERE table_schema = 'cibertronia_salud';
