<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::table('historias_clinicas', function (Blueprint $table) {
            // CIE-10
            $table->unsignedBigInteger('cie10_id')->nullable()->after('examenes_solicitados');
            $table->string('cie10_codigo_texto', 20)->nullable()->after('cie10_id')->comment('Copia del código para búsqueda rápida');
            $table->enum('tipo_diagnostico', ['presuntivo', 'definitivo'])->default('presuntivo')->after('cie10_codigo_texto');

            // SOAP
            $table->json('notas_soap')->nullable()->after('tipo_diagnostico')
                ->comment('{"subjetivo":"...","objetivo":"...","analisis":"...","plan":"..."}');

            // Examen físico por sistemas
            $table->json('examen_fisico')->nullable()->after('notas_soap')
                ->comment('[{"sistema":"cardiovascular","hallazgos":"normal"}]');

            // Bloqueo de edición después de 24h
            $table->boolean('bloqueada_para_edicion')->default(false)->after('examen_fisico');
            $table->timestamp('fecha_bloqueo')->nullable()->after('bloqueada_para_edicion');

            // Auditoría
            $table->json('auditoria')->nullable()->after('fecha_bloqueo')
                ->comment('{"creado_por":"...","creado_fecha":"...","editado_por":"...","editado_fecha":"..."}');

            // FK a catálogo CIE-10
            $table->foreign('cie10_id')
                ->references('id')->on('cie10_catalog')
                ->onDelete('set null');

            $table->index('cie10_id');
            $table->index('tipo_diagnostico');
        });
    }

    public function down(): void
    {
        Schema::table('historias_clinicas', function (Blueprint $table) {
            $table->dropForeign(['cie10_id']);
            $table->dropIndex(['cie10_id']);
            $table->dropIndex(['tipo_diagnostico']);
            $table->dropColumn([
                'cie10_id', 'cie10_codigo_texto', 'tipo_diagnostico',
                'notas_soap', 'examen_fisico',
                'bloqueada_para_edicion', 'fecha_bloqueo', 'auditoria',
            ]);
        });
    }
};
