<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use App\Models\Categoria;
use App\Models\Estudio;
use App\Models\Cita;
use App\Models\Paciente;
use App\Models\Usuario;
use App\Models\HistoriaClinica;
use App\Models\Cie10Catalog;
use App\Models\RecetaMedica;
use App\Models\MedicamentoReceta;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

// Importar el seeder CIE10
use Database\Seeders\Cie10CatalogSeeder;

class DemoDataSeeder extends Seeder
{
    public function run()
    {
        Schema::disableForeignKeyConstraints();
        
        $password = Hash::make('password123');

        // Médico de Ejemplo
        $medico = Usuario::firstOrCreate([
            'email' => 'medico@cibertroniasalud.com', // Usar el email que ya funciona
        ], [
            'nombre' => 'Doctor',
            'apellido' => 'Ejemplo',
            'password' => $password,
            'rol' => 'medico',
            'especialidad' => 'Medicina General',
            'telefono' => '555-0101',
            'estado' => 'activo'
        ]);

        // Verificar CIE-10
        if (Cie10Catalog::count() == 0) {
            $this->command->info('Sembrando CIE-10...');
            $this->call(Cie10CatalogSeeder::class);
        }

        // 1. Crear Categorías
        $categoriasData = [
            ['nombre' => 'Consulta General', 'icono' => '🩺', 'color' => '#3B82F6', 'orden' => 1],
            ['nombre' => 'Odontología', 'icono' => '🦷', 'color' => '#10B981', 'orden' => 2],
            ['nombre' => 'Laboratorio', 'icono' => '🔬', 'color' => '#8B5CF6', 'orden' => 3],
            ['nombre' => 'Rayos X', 'icono' => '🦴', 'color' => '#F59E0B', 'orden' => 4],
            ['nombre' => 'Pediatría', 'icono' => '👶', 'color' => '#EC4899', 'orden' => 5],
        ];

        foreach ($categoriasData as $cat) {
            Categoria::firstOrCreate(['nombre' => $cat['nombre']], $cat);
        }
        $this->command->info('Categorías creadas.');

        // 2. Crear Estudios
        $estudiosData = [
            ['nombre' => 'Consulta Médica General', 'categoria' => 'Consulta General', 'codigo' => 'CONS-GEN', 'precio' => 150.00],
            ['nombre' => 'Limpieza Dental', 'categoria' => 'Odontología', 'codigo' => 'ODONT-LMP', 'precio' => 250.00],
            ['nombre' => 'Hemograma Completo', 'categoria' => 'Laboratorio', 'codigo' => 'LAB-HEMO', 'precio' => 80.00],
            ['nombre' => 'Radiografía de Tórax', 'categoria' => 'Rayos X', 'codigo' => 'RX-TOR', 'precio' => 300.00],
            ['nombre' => 'Control Niño Sano', 'categoria' => 'Pediatría', 'codigo' => 'PED-CTRL', 'precio' => 180.00],
        ];

        foreach ($estudiosData as $est) {
            Estudio::updateOrCreate(['codigo' => $est['codigo']], $est);
        }
        $this->command->info('Estudios creados.');

        // 3. Crear 5 Pacientes de Ejemplo
        $pacientesData = [
            ['nombre' => 'Carlos', 'apellido' => 'Vargas', 'dni' => '10000001', 'sexo' => 'M', 'nacimiento' => '1980-05-15'],
            ['nombre' => 'Lucía', 'apellido' => 'Méndez', 'dni' => '10000002', 'sexo' => 'F', 'nacimiento' => '1992-08-20'],
            ['nombre' => 'Jorge', 'apellido' => 'Salazar', 'dni' => '10000003', 'sexo' => 'M', 'nacimiento' => '1955-12-10'],
            ['nombre' => 'Ana', 'apellido' => 'Torres', 'dni' => '10000004', 'sexo' => 'F', 'nacimiento' => '2015-03-30'],
            ['nombre' => 'Miguel', 'apellido' => 'Rios', 'dni' => '10000005', 'sexo' => 'M', 'nacimiento' => '1998-07-22'],
        ];

        foreach ($pacientesData as $p) {
            try {
                Paciente::firstOrCreate(
                    ['numero_documento' => $p['dni']],
                    [
                        'numero_historia' => 'HC-' . $p['dni'],
                        'nombre' => $p['nombre'],
                        'apellido' => $p['apellido'],
                        'fecha_nacimiento' => $p['nacimiento'],
                        'sexo' => $p['sexo'],
                        'tipo_documento' => 'DNI',
                        'telefono' => '555-0000',
                        'email' => strtolower("{$p['nombre']}.{$p['apellido']}@test.com"),
                        'estado' => 'activo',
                        'registrado_por' => 1
                    ]
                );
            } catch (\Exception $e) {
                // Ignore duplicates
            }
        }
        $this->command->info('Pacientes de ejemplo creados.');

        // 4. Crear Citas y H.C.
        $pacientes = Paciente::whereIn('numero_documento', collect($pacientesData)->pluck('dni'))->get();
        $diagnosticos = Cie10Catalog::inRandomOrder()->limit(5)->get();

        if ($diagnosticos->isEmpty()) {
            $this->command->error('No se encontraron diagnósticos CIE-10. Asegúrate de correr Cie10CatalogSeeder.');
            return;
        }

        foreach ($pacientes as $index => $paciente) {
            // CITA PASADA CON HISTORIA
            $fechaPasada = Carbon::now()->subDays(rand(1, 30))->setHour(10)->setMinute(0);
            
            // Check if cita exists
            $citaPasada = Cita::firstOrCreate([
                'paciente_id' => $paciente->id,
                'fecha_cita' => $fechaPasada
            ], [
                'medico_id' => $medico->id,
                'duracion_minutos' => 30,
                'tipo_cita' => 'consulta_general',
                'motivo_consulta' => 'Dolor abdominal persistente',
                'estado' => 'completada',
                'sala_consultorio' => 'Consultorio 1',
                'creada_por' => 1
            ]);

            // Crear Historia Clínica (Check if exists first)
            $historiaExiste = HistoriaClinica::where('cita_id', $citaPasada->id)->exists();
            
            if (!$historiaExiste) {
                try {
                    $cie10 = $diagnosticos->random();
                    $historia = HistoriaClinica::create([
                        'paciente_id' => $paciente->id,
                        'medico_id' => $medico->id,
                        'cita_id' => $citaPasada->id,
                        'fecha_consulta' => $fechaPasada,
                        'motivo_consulta' => 'Dolor abdominal',
                        'notas_soap' => [
                            'subjetivo' => 'Paciente refiere dolor abdominal n tipo cólico de 3 días de evolución.',
                            'objetivo' => 'Abdomen blando, depresible, doloroso a la palpación en flanco derecho.',
                            'analisis' => 'Probable gastroenteritis viral vs inicio de apendicitis.',
                            'plan' => 'Reposo, hidratación, analgésicos y observación.'
                        ],
                        'examen_fisico' => [
                            ['sistema' => 'Abdomen', 'hallazgos' => 'Dolor focalizado']
                        ],
                        'cie10_id' => $cie10->id,
                        'cie10_codigo_texto' => $cie10->codigo,
                        'tipo_diagnostico' => 'presuntivo'
                    ]);

                    // Crear Receta
                    $receta = RecetaMedica::create([
                        'historia_clinica_id' => $historia->id,
                        'paciente_id' => $paciente->id,
                        'medico_id' => $medico->id,
                        'fecha_emision' => $fechaPasada,
                        'estado' => 'activa'
                    ]);
                    
                    MedicamentoReceta::create([
                        'receta_id' => $receta->id,
                        'nombre_medicamento' => 'Paracetamol',
                        'dosis' => '500mg',
                        'frecuencia' => 'Cada 8 horas',
                        'duracion_dias' => 3,
                        'cantidad_total' => 9
                    ]);
                } catch (\Exception $e) {
                    $this->command->error("Error creando historia para {$paciente->nombre}: " . $e->getMessage());
                }
            }

            // CITA FUTURA (Pendiente) - check duplicate
            $fechaFutura = Carbon::now()->addDays(rand(1, 7))->setHour(9)->setMinute(30);
            $existeFutura = Cita::where('paciente_id', $paciente->id)->where('estado', 'programada')->exists();
            
            if (!$existeFutura) {
                Cita::create([
                    'paciente_id' => $paciente->id,
                    'medico_id' => $medico->id,
                    'fecha_cita' => $fechaFutura,
                    'duracion_minutos' => 30,
                    'tipo_cita' => 'control',
                    'motivo_consulta' => 'Control de seguimiento',
                    'estado' => 'programada',
                    'creada_por' => 1
                ]);
            }
        }

        $this->command->info('Historias Clínicas y Citas generadas exitosamente.');
        Schema::enableForeignKeyConstraints();
    }
}
