<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

class HistorialEjemploSeeder extends Seeder
{
    public function run(): void
    {
        $paciente = DB::table('pacientes')->where('numero_documento', '12345678')->first();
        $medico   = DB::table('usuarios')->where('email', 'pukriento@gmail.com')->first();

        if (!$paciente || !$medico) {
            $this->command->error('No se encontró el paciente (CI: 12345678) o el médico (pukriento@gmail.com).');
            return;
        }

        $pid = $paciente->id;
        $mid = $medico->id;
        $cie10 = DB::table('cie10_catalog')->where('codigo', 'LIKE', 'G43%')->first();
        $cie10Id = $cie10 ? $cie10->id : null;

        // ─── CITA 1: Primera consulta (hace 3 meses) ─────────────────────
        $c1 = DB::table('citas')->insertGetId([
            'paciente_id'      => $pid,
            'medico_id'        => $mid,
            'fecha_cita'       => Carbon::now()->subMonths(3)->setTime(9, 0),
            'duracion_minutos' => 30,
            'tipo_cita'        => 'consulta_general',
            'motivo_consulta'  => 'Cefalea intensa y mareos recurrentes',
            'estado'           => 'completada',
            'observaciones'    => 'Primera consulta. Paciente refiere cefalea de 3 días de evolución.',
            'creada_por'       => $mid,
            'fecha_creacion'   => Carbon::now()->subMonths(3),
            'fecha_actualizacion' => Carbon::now()->subMonths(3),
        ]);

        $h1 = DB::table('historias_clinicas')->insertGetId([
            'cita_id'              => $c1,
            'paciente_id'          => $pid,
            'medico_id'            => $mid,
            'fecha_consulta'       => Carbon::now()->subMonths(3),
            'motivo_consulta'      => 'Cefalea intensa y mareos recurrentes',
            'presion_arterial'     => '140/90',
            'frecuencia_cardiaca'  => 88,
            'temperatura'          => 36.8,
            'frecuencia_respiratoria' => 18,
            'peso'                 => 82.5,
            'altura'               => 1.72,
            'saturacion_oxigeno'   => 97,
            'sintomas'             => 'Cefalea pulsátil intensa, mareos, fotofobia, náuseas. Inicio hace 3 días.',
            'diagnostico'          => 'Migraña con aura. Hipertensión arterial estadio 1.',
            'cie10_id'             => $cie10Id,
            'cie10_codigo_texto'   => 'G43',
            'tipo_diagnostico'     => 'presuntivo',
            'notas_soap'           => json_encode([
                'subjetivo' => 'Paciente masculino de 35 años refiere cefalea pulsátil intensa de 3 días de evolución, localizada en región frontal y temporal bilateral. Fotofobia y náuseas. Antecedente familiar de migraña (madre). Trabajo bajo estrés considerable.',
                'objetivo'  => 'PA: 140/90 mmHg. FC: 88 lpm. T°: 36.8°C. FR: 18 rpm. SpO2: 97%. Peso: 82.5 kg. Talla: 1.72 m. IMC: 27.9. Fondo de ojo: leve papiledema bilateral. Neurológico: sin déficit focal.',
                'analisis'  => 'Cuadro compatible con migraña con aura asociado a HTA estadio 1. Se descarta proceso orgánico intracraneal. Factores desencadenantes: estrés laboral, privación del sueño.',
                'plan'      => 'Sumatriptán 50 mg para episodios agudos. Propranolol 40 mg profilaxis. Losartán 50 mg para HTA. Se solicita BH, QS, Rx tórax. Próximo control en 30 días.',
            ]),
            'examen_fisico'        => json_encode([
                ['sistema' => 'Neurológico',     'hallazgos' => 'Sin déficit focal. ROT presentes y simétricos.'],
                ['sistema' => 'Cardiovascular',  'hallazgos' => 'Ruidos cardíacos rítmicos, sin soplos.'],
                ['sistema' => 'Cabeza y Cuello', 'hallazgos' => 'Dolor a la palpación de senos frontales. Sin rigidez de nuca.'],
                ['sistema' => 'Respiratorio',    'hallazgos' => 'Murmullo vesicular conservado.'],
            ]),
            'tratamiento'          => 'Sumatriptán 50 mg c/necesidad. Propranolol 40 mg c/12h. Losartán 50 mg c/24h.',
            'recomendaciones'      => 'Dieta hiposódica. Actividad física 30 min/día. Llevar diario de cefaleas.',
            'proxima_cita'         => Carbon::now()->subMonths(2)->format('Y-m-d'),
            'auditoria'            => json_encode(['creado_por' => $mid, 'creado_fecha' => Carbon::now()->subMonths(3)->toIso8601String()]),
            'fecha_creacion'       => Carbon::now()->subMonths(3),
            'fecha_actualizacion'  => Carbon::now()->subMonths(3),
        ]);

        // Receta 1
        $r1 = DB::table('recetas_medicas')->insertGetId([
            'historia_clinica_id'     => $h1,
            'paciente_id'             => $pid,
            'medico_id'               => $mid,
            'fecha_emision'           => Carbon::now()->subMonths(3)->format('Y-m-d'),
            'vigencia_dias'           => 30,
            'instrucciones_generales' => 'Tomar con abundante agua. No suspender sin consultar. Evitar alcohol.',
            'estado'                  => 'activa',
            'fecha_creacion'          => Carbon::now()->subMonths(3),
            'fecha_actualizacion'     => Carbon::now()->subMonths(3),
        ]);
        DB::table('medicamentos_receta')->insert([
            ['receta_id' => $r1, 'nombre_medicamento' => 'Sumatriptán',  'dosis' => '50 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Al inicio del episodio',     'duracion_dias' => 30, 'cantidad_total' => 6,  'instrucciones_especiales' => 'Máx 2 tabletas por ataque. No más de 2 ataques/semana.', 'fecha_creacion' => Carbon::now()->subMonths(3)],
            ['receta_id' => $r1, 'nombre_medicamento' => 'Propranolol',  'dosis' => '40 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Cada 12 horas',              'duracion_dias' => 30, 'cantidad_total' => 60, 'instrucciones_especiales' => 'No suspender bruscamente. Tomar con alimentos.',         'fecha_creacion' => Carbon::now()->subMonths(3)],
            ['receta_id' => $r1, 'nombre_medicamento' => 'Losartán',     'dosis' => '50 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Cada 24 horas (mañana)',     'duracion_dias' => 30, 'cantidad_total' => 30, 'instrucciones_especiales' => 'Controlar PA semanalmente.',                                  'fecha_creacion' => Carbon::now()->subMonths(3)],
        ]);

        // Exámenes cita 1
        DB::table('examenes_medicos')->insert([
            ['paciente_id' => $pid, 'medico_solicitante_id' => $mid, 'historia_clinica_id' => $h1, 'tipo_examen' => 'laboratorio',  'descripcion' => 'Biometría Hemática',   'fecha_solicitud' => Carbon::now()->subMonths(3)->format('Y-m-d'), 'fecha_realizacion' => Carbon::now()->subMonths(3)->addDays(2)->format('Y-m-d'), 'laboratorio_centro' => 'Lab. Cibertronia', 'resultados' => 'Hemoglobina: 14.5 g/dL. Leucocitos: 7,200/mm³. Plaquetas: 245,000/mm³. Normales.', 'estado' => 'completado', 'observaciones' => 'Sin alteraciones.', 'fecha_creacion' => Carbon::now()->subMonths(3), 'fecha_actualizacion' => Carbon::now()->subMonths(3)->addDays(2)],
            ['paciente_id' => $pid, 'medico_solicitante_id' => $mid, 'historia_clinica_id' => $h1, 'tipo_examen' => 'laboratorio',  'descripcion' => 'Química Sanguínea',    'fecha_solicitud' => Carbon::now()->subMonths(3)->format('Y-m-d'), 'fecha_realizacion' => Carbon::now()->subMonths(3)->addDays(2)->format('Y-m-d'), 'laboratorio_centro' => 'Lab. Cibertronia', 'resultados' => 'Glucosa: 105 mg/dL. Colesterol: 210 mg/dL. TG: 185 mg/dL (↑). Creatinina: 0.9 mg/dL.', 'estado' => 'completado', 'observaciones' => 'Leve dislipidemia. Glucosa en límite.', 'fecha_creacion' => Carbon::now()->subMonths(3), 'fecha_actualizacion' => Carbon::now()->subMonths(3)->addDays(2)],
            ['paciente_id' => $pid, 'medico_solicitante_id' => $mid, 'historia_clinica_id' => $h1, 'tipo_examen' => 'imagenologia', 'descripcion' => 'Radiografía de Tórax', 'fecha_solicitud' => Carbon::now()->subMonths(3)->format('Y-m-d'), 'fecha_realizacion' => Carbon::now()->subMonths(3)->addDay()->format('Y-m-d'),   'laboratorio_centro' => 'Imagen. Cibertronia', 'resultados' => 'Silueta cardíaca normal. Campos pulmonares sin infiltrados. Senos costofrénicos libres.', 'estado' => 'completado', 'observaciones' => 'Estudio normal.', 'fecha_creacion' => Carbon::now()->subMonths(3), 'fecha_actualizacion' => Carbon::now()->subMonths(3)->addDay()],
        ]);

        // ─── CITA 2: 1er control (hace 2 meses) ──────────────────────────
        $c2 = DB::table('citas')->insertGetId([
            'paciente_id'      => $pid, 'medico_id' => $mid,
            'fecha_cita'       => Carbon::now()->subMonths(2)->setTime(10, 30),
            'duracion_minutos' => 20, 'tipo_cita' => 'control',
            'motivo_consulta'  => 'Control de migraña e HTA. Revisión de resultados.',
            'estado'           => 'completada', 'creada_por' => $mid,
            'fecha_creacion'   => Carbon::now()->subMonths(2),
            'fecha_actualizacion' => Carbon::now()->subMonths(2),
        ]);

        $h2 = DB::table('historias_clinicas')->insertGetId([
            'cita_id' => $c2, 'paciente_id' => $pid, 'medico_id' => $mid,
            'fecha_consulta'       => Carbon::now()->subMonths(2),
            'motivo_consulta'      => 'Control de migraña e hipertensión. Revisión de resultados de laboratorio.',
            'presion_arterial'     => '130/85', 'frecuencia_cardiaca' => 72, 'temperatura' => 36.5,
            'frecuencia_respiratoria' => 16, 'peso' => 81.0, 'altura' => 1.72, 'saturacion_oxigeno' => 98,
            'sintomas'    => '2 episodios de migraña en el mes, menos intensos. Tolera bien el tratamiento.',
            'diagnostico' => 'Mejoría clínica de migraña con tratamiento. HTA en control parcial.',
            'cie10_id' => $cie10Id, 'cie10_codigo_texto' => 'G43', 'tipo_diagnostico' => 'definitivo',
            'notas_soap' => json_encode([
                'subjetivo' => 'Solo 2 episodios de migraña vs 8-10 anteriores. Menor intensidad (EVA 5/10 vs 8/10). Tolera bien Propranolol. Realizó cambios en estilo de vida: redujo café, mejor sueño.',
                'objetivo'  => 'PA: 130/85 mmHg (mejoría). FC: 72 lpm. Peso: 81 kg (bajó 1.5 kg). Laboratorios revisados: Colesterol 210, TG 185 (leve dislipidemia), glucosa 105.',
                'analisis'  => 'Excelente respuesta al Propranolol. HTA parcialmente controlada. Dislipidemia leve.',
                'plan'      => 'Continuar Propranolol + Losartán. Agregar Rosuvastatina 10 mg/noche. Dieta mediterránea. Próximo control en 60 días con perfil glucémico.',
            ]),
            'examen_fisico' => json_encode([
                ['sistema' => 'Cardiovascular', 'hallazgos' => 'PA 130/85. FC 72 regular. Sin signos de ICC.'],
                ['sistema' => 'Neurológico',    'hallazgos' => 'Cefalea ausente. Sin déficit focal.'],
            ]),
            'tratamiento'  => 'Propranolol 40mg c/12h. Losartán 50mg c/24h. Rosuvastatina 10mg c/noche (NUEVO).',
            'recomendaciones' => 'Dieta mediterránea. 45 min actividad física/día. Meta peso: 78 kg.',
            'proxima_cita' => Carbon::now()->subMonths(1)->addDays(5)->format('Y-m-d'),
            'auditoria'    => json_encode(['creado_por' => $mid, 'creado_fecha' => Carbon::now()->subMonths(2)->toIso8601String()]),
            'fecha_creacion' => Carbon::now()->subMonths(2), 'fecha_actualizacion' => Carbon::now()->subMonths(2),
        ]);

        $r2 = DB::table('recetas_medicas')->insertGetId([
            'historia_clinica_id' => $h2, 'paciente_id' => $pid, 'medico_id' => $mid,
            'fecha_emision'           => Carbon::now()->subMonths(2)->format('Y-m-d'),
            'vigencia_dias'           => 60,
            'instrucciones_generales' => 'Agregar Rosuvastatina en la noche con cena. No consumir toronja.',
            'estado' => 'activa',
            'fecha_creacion' => Carbon::now()->subMonths(2), 'fecha_actualizacion' => Carbon::now()->subMonths(2),
        ]);
        DB::table('medicamentos_receta')->insert([
            ['receta_id' => $r2, 'nombre_medicamento' => 'Propranolol',   'dosis' => '40 mg',  'via_administracion' => 'oral', 'frecuencia' => 'Cada 12 horas',       'duracion_dias' => 60, 'cantidad_total' => 120, 'instrucciones_especiales' => 'Continuar sin interrupciones.', 'fecha_creacion' => Carbon::now()->subMonths(2)],
            ['receta_id' => $r2, 'nombre_medicamento' => 'Losartán',      'dosis' => '50 mg',  'via_administracion' => 'oral', 'frecuencia' => 'Cada 24 horas',       'duracion_dias' => 60, 'cantidad_total' => 60,  'instrucciones_especiales' => 'Con o sin alimentos.',           'fecha_creacion' => Carbon::now()->subMonths(2)],
            ['receta_id' => $r2, 'nombre_medicamento' => 'Rosuvastatina', 'dosis' => '10 mg',  'via_administracion' => 'oral', 'frecuencia' => 'Cada noche con cena', 'duracion_dias' => 60, 'cantidad_total' => 60,  'instrucciones_especiales' => 'EVITAR toronja. Reportar dolor muscular.', 'fecha_creacion' => Carbon::now()->subMonths(2)],
        ]);

        // ─── CITA 3: Control reciente (hace 2 semanas) ───────────────────
        $c3 = DB::table('citas')->insertGetId([
            'paciente_id'      => $pid, 'medico_id' => $mid,
            'fecha_cita'       => Carbon::now()->subWeeks(2)->setTime(8, 30),
            'duracion_minutos' => 30, 'tipo_cita' => 'control',
            'motivo_consulta'  => 'Control semestral. Evaluación integral.',
            'estado'           => 'completada', 'creada_por' => $mid,
            'fecha_creacion'   => Carbon::now()->subWeeks(2),
            'fecha_actualizacion' => Carbon::now()->subWeeks(2),
        ]);

        $h3 = DB::table('historias_clinicas')->insertGetId([
            'cita_id' => $c3, 'paciente_id' => $pid, 'medico_id' => $mid,
            'fecha_consulta'       => Carbon::now()->subWeeks(2),
            'motivo_consulta'      => 'Control semestral. Evaluación de HTA y migraña. Revisión de perfil glucémico.',
            'presion_arterial'     => '125/80', 'frecuencia_cardiaca' => 68, 'temperatura' => 36.6,
            'frecuencia_respiratoria' => 16, 'peso' => 79.5, 'altura' => 1.72, 'saturacion_oxigeno' => 98,
            'sintomas'    => '1 episodio de migraña leve en 2 meses. PA bien controlada. Perdió 3 kg.',
            'diagnostico' => 'HTA controlada. Migraña en remisión. Dislipidemia en mejoría.',
            'cie10_id' => $cie10Id, 'cie10_codigo_texto' => 'I10', 'tipo_diagnostico' => 'definitivo',
            'notas_soap' => json_encode([
                'subjetivo' => 'Excelente evolución. 1 episodio de migraña muy leve en 2 meses. Adherencia completa al tratamiento. Camina 5 km diarios, dieta mediterránea. Perdió 3 kg. Duerme mejor.',
                'objetivo'  => 'PA: 125/80 mmHg (excelente). FC: 68 lpm. Peso: 79.5 kg (↓3 kg). IMC: 26.9. Labs.: Colesterol 178 mg/dL (↓), TG 145 mg/dL (↓), Glucosa: 98 mg/dL, HbA1c: 5.4% (normal).',
                'analisis'  => 'Magnífica respuesta y adherencia. Migraña en remisión funcional. HTA controlada. Dislipidemia resuelta.',
                'plan'      => 'Mantener tratamiento actual. Valorar reducción de Propranolol en 3 meses. Perfil lipídico en 3 meses. Se solicita EGO de control. Próximo control en 3 meses.',
            ]),
            'examen_fisico' => json_encode([
                ['sistema' => 'General',         'hallazgos' => 'Buen estado general. Bien hidratado, normopeso.'],
                ['sistema' => 'Cardiovascular',  'hallazgos' => 'PA 125/80 bilateral. FC 68 rítmica. Sin edemas.'],
                ['sistema' => 'Neurológico',     'hallazgos' => 'Sin signos de focalización. Cefalea ausente.'],
                ['sistema' => 'Abdomen',         'hallazgos' => 'Blando, depresible, no doloroso. Sin visceromegalias.'],
            ]),
            'tratamiento'  => 'Propranolol 40 mg c/12h. Losartán 50 mg c/24h. Rosuvastatina 10 mg c/noche.',
            'recomendaciones' => '¡Felicitaciones por el progreso! Continuar con el excelente estilo de vida. Meta peso: 77 kg.',
            'proxima_cita' => Carbon::now()->addMonths(3)->format('Y-m-d'),
            'auditoria'    => json_encode(['creado_por' => $mid, 'creado_fecha' => Carbon::now()->subWeeks(2)->toIso8601String()]),
            'fecha_creacion' => Carbon::now()->subWeeks(2), 'fecha_actualizacion' => Carbon::now()->subWeeks(2),
        ]);

        DB::table('examenes_medicos')->insert([
            ['paciente_id' => $pid, 'medico_solicitante_id' => $mid, 'historia_clinica_id' => $h3,
             'tipo_examen' => 'laboratorio', 'descripcion' => 'Examen General de Orina',
             'fecha_solicitud' => Carbon::now()->subWeeks(2)->format('Y-m-d'),
             'fecha_realizacion' => null, 'laboratorio_centro' => 'Lab. Cibertronia',
             'resultados' => null, 'estado' => 'solicitado',
             'observaciones' => 'Pendiente realización por parte del paciente.',
             'fecha_creacion' => Carbon::now()->subWeeks(2), 'fecha_actualizacion' => Carbon::now()->subWeeks(2)],
        ]);

        $r3 = DB::table('recetas_medicas')->insertGetId([
            'historia_clinica_id' => $h3, 'paciente_id' => $pid, 'medico_id' => $mid,
            'fecha_emision'           => Carbon::now()->subWeeks(2)->format('Y-m-d'),
            'vigencia_dias'           => 90,
            'instrucciones_generales' => 'Traer resultados de EGO en próxima consulta.',
            'estado' => 'activa',
            'fecha_creacion' => Carbon::now()->subWeeks(2), 'fecha_actualizacion' => Carbon::now()->subWeeks(2),
        ]);
        DB::table('medicamentos_receta')->insert([
            ['receta_id' => $r3, 'nombre_medicamento' => 'Propranolol',   'dosis' => '40 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Cada 12 horas',       'duracion_dias' => 90, 'cantidad_total' => 180, 'instrucciones_especiales' => 'Se evaluará reducción en próxima consulta.', 'fecha_creacion' => Carbon::now()->subWeeks(2)],
            ['receta_id' => $r3, 'nombre_medicamento' => 'Losartán',      'dosis' => '50 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Cada 24 horas',       'duracion_dias' => 90, 'cantidad_total' => 90,  'instrucciones_especiales' => null, 'fecha_creacion' => Carbon::now()->subWeeks(2)],
            ['receta_id' => $r3, 'nombre_medicamento' => 'Rosuvastatina', 'dosis' => '10 mg', 'via_administracion' => 'oral', 'frecuencia' => 'Cada noche con cena', 'duracion_dias' => 90, 'cantidad_total' => 90,  'instrucciones_especiales' => 'Repetir perfil lipídico en 3 meses.', 'fecha_creacion' => Carbon::now()->subWeeks(2)],
        ]);

        // ─── CITA FUTURA programada ───────────────────────────────────────
        DB::table('citas')->insertGetId([
            'paciente_id'      => $pid, 'medico_id' => $mid,
            'fecha_cita'       => Carbon::now()->addMonths(3)->setTime(9, 0),
            'duracion_minutos' => 30, 'tipo_cita' => 'control',
            'motivo_consulta'  => 'Control médico trimestral. HTA y migraña.',
            'estado'           => 'programada', 'creada_por' => $mid,
            'observaciones'    => 'Traer resultados EGO, perfil lipídico y registro de PA.',
            'fecha_creacion'   => Carbon::now()->subWeeks(2),
            'fecha_actualizacion' => Carbon::now()->subWeeks(2),
        ]);

        $this->command->info('✅ Historial clínico completo creado para Carlos Fernández (CI: 12345678):');
        $this->command->info('   - 3 Historias clínicas + SOAP + Examen físico');
        $this->command->info('   - 3 Recetas médicas con 9 medicamentos en total');
        $this->command->info('   - 4 Exámenes (3 completados + 1 pendiente)');
        $this->command->info('   - 1 Cita futura programada (3 meses)');
    }
}
