<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class PacientesEjemplo extends Seeder
{
    public function run(): void
    {
        // Obtener el admin para asignarlo como médico y registrador
        $admin = DB::table('usuarios')->where('email', 'admin@cibertronia.com')->first();
        if (!$admin) {
            $this->command->error('No se encontró el usuario admin. Ejecuta primero DatosIniciales.');
            return;
        }

        // Obtener algunos exámenes existentes
        $hemograma   = DB::table('estudios')->where('codigo', 'HEM001')->first();
        $glicemia    = DB::table('estudios')->where('codigo', 'BIO001')->first();
        $lipidico    = DB::table('estudios')->where('codigo', 'BIO002')->first();
        $vih         = DB::table('estudios')->where('codigo', 'INM001')->first();
        $urianalisis = DB::table('estudios')->where('codigo', 'URI001')->first();

        // =============================================
        // PACIENTES
        // =============================================
        $pacientes = [
            [
                'ci'             => '12345678',
                'nombre'         => 'María',
                'apellido'       => 'García López',
                'celular'        => '70012345',
                'sexo'           => 'Femenino',
                'direccion'      => 'Av. Siempre Viva 742, La Paz',
                'fecha_nacimiento'=> '1985-03-15',
                'email'          => 'maria.garcia@gmail.com',
                'notas'          => 'Paciente con antecedentes de anemia.',
                'examen_id'      => $hemograma?->id ?? 1,
                'precio'         => 45.00,
                'metodo_pago'    => 'Efectivo',
                'estado_pago'    => 'pagado',   // ← PAGADO
                'estado'         => 'activo',
            ],
            [
                'ci'             => '23456789',
                'nombre'         => 'Juan',
                'apellido'       => 'Mamani Quispe',
                'celular'        => '70023456',
                'sexo'           => 'Masculino',
                'direccion'      => 'Calle Bolívar 120, Cochabamba',
                'fecha_nacimiento'=> '1978-07-22',
                'email'          => 'juan.mamani@hotmail.com',
                'notas'          => 'Control de hipertensión.',
                'examen_id'      => $lipidico?->id ?? 2,
                'precio'         => 60.00,
                'metodo_pago'    => 'QR',
                'estado_pago'    => 'pendiente', // ← PENDIENTE
                'estado'         => 'activo',
            ],
            [
                'ci'             => '34567890',
                'nombre'         => 'Ana',
                'apellido'       => 'Flores Vda. de Ramos',
                'celular'        => '71034567',
                'sexo'           => 'Femenino',
                'direccion'      => 'Av. América 45, Santa Cruz',
                'fecha_nacimiento'=> '1992-11-08',
                'email'          => 'ana.flores@gmail.com',
                'notas'          => 'Primera visita. Sin antecedentes.',
                'examen_id'      => $glicemia?->id ?? 1,
                'precio'         => 20.00,
                'metodo_pago'    => 'Efectivo',
                'estado_pago'    => 'pagado',   // ← PAGADO
                'estado'         => 'activo',
            ],
            [
                'ci'             => '45678901',
                'nombre'         => 'Carlos',
                'apellido'       => 'Torrez Blanco',
                'celular'        => '72045678',
                'sexo'           => 'Masculino',
                'direccion'      => 'Barrio Recoleta Calle 3, Sucre',
                'fecha_nacimiento'=> '1965-01-30',
                'email'          => null,
                'notas'          => 'Derivado por médico general. Sospecha VIH.',
                'examen_id'      => $vih?->id ?? 1,
                'precio'         => 50.00,
                'metodo_pago'    => 'Efectivo',
                'estado_pago'    => 'pendiente', // ← PENDIENTE
                'estado'         => 'activo',
            ],
            [
                'ci'             => '56789012',
                'nombre'         => 'Sofía',
                'apellido'       => 'Condori Huanca',
                'celular'        => '73056789',
                'sexo'           => 'Femenino',
                'direccion'      => 'Urb. Las Rosas Mz. 5, El Alto',
                'fecha_nacimiento'=> '2000-05-17',
                'email'          => 'sofia.condori@yahoo.com',
                'notas'          => 'Examen rutinario pre-operatorio.',
                'examen_id'      => $urianalisis?->id ?? 1,
                'precio'         => 25.00,
                'metodo_pago'    => 'Débito',
                'estado_pago'    => 'pagado',   // ← PAGADO
                'estado'         => 'activo',
            ],
        ];

        $pacienteIds = [];
        foreach ($pacientes as $p) {
            if (!DB::table('pacientes')->where('ci', $p['ci'])->exists()) {
                $pacienteIds[$p['ci']] = DB::table('pacientes')->insertGetId(array_merge($p, [
                    'created_at' => now(),
                    'updated_at' => now(),
                ]));
            } else {
                $pacienteIds[$p['ci']] = DB::table('pacientes')->where('ci', $p['ci'])->value('id');
            }
        }

        // =============================================
        // CITAS  (una cita por paciente, reflejando su estado de pago)
        // =============================================
        $citasData = [
            [
                'ci'            => '12345678',
                'fecha_cita'    => now()->subDays(2)->setTime(9, 0),
                'tipo_cita'     => 'examen',
                'motivo_consulta'=> 'Control de hemograma. Paciente con antecedentes de anemia.',
                'estado'        => 'completada',
                'pagado'        => true,   // ← refleja estado pagado
            ],
            [
                'ci'            => '23456789',
                'fecha_cita'    => now()->addDays(1)->setTime(10, 30),
                'tipo_cita'     => 'consulta_general',
                'motivo_consulta'=> 'Control de perfil lipídico. Seguimiento hipertensión.',
                'estado'        => 'programada',
                'pagado'        => false,  // ← refleja estado pendiente
            ],
            [
                'ci'            => '34567890',
                'fecha_cita'    => now()->subDays(5)->setTime(11, 0),
                'tipo_cita'     => 'examen',
                'motivo_consulta'=> 'Glicemia en ayunas. Primera visita.',
                'estado'        => 'completada',
                'pagado'        => true,   // ← refleja estado pagado
            ],
            [
                'ci'            => '45678901',
                'fecha_cita'    => now()->addDays(3)->setTime(8, 0),
                'tipo_cita'     => 'consulta_especializada',
                'motivo_consulta'=> 'Prueba de VIH. Derivado por médico general.',
                'estado'        => 'confirmada',
                'pagado'        => false,  // ← refleja estado pendiente
            ],
            [
                'ci'            => '56789012',
                'fecha_cita'    => now()->subDays(1)->setTime(14, 30),
                'tipo_cita'     => 'examen',
                'motivo_consulta'=> 'Urianálisis completo. Pre-operatorio.',
                'estado'        => 'completada',
                'pagado'        => true,   // ← refleja estado pagado
            ],
        ];

        foreach ($citasData as $c) {
            $pacienteId = $pacienteIds[$c['ci']] ?? null;
            if (!$pacienteId) continue;

            DB::table('citas')->insert([
                'paciente_id'    => $pacienteId,
                'medico_id'      => $admin->id,
                'fecha_cita'     => $c['fecha_cita'],
                'duracion_minutos'=> 30,
                'tipo_cita'      => $c['tipo_cita'],
                'motivo_consulta'=> $c['motivo_consulta'],
                'estado'         => $c['estado'],
                'pagado'         => $c['pagado'],
                'sala_consultorio'=> 'Sala 1',
                'creada_por'     => $admin->id,
                'fecha_creacion' => now(),
                'fecha_actualizacion' => now(),
            ]);
        }

        $this->command->info('✅ 5 pacientes de ejemplo creados con sus citas (3 pagadas, 2 pendientes).');
    }
}
