import { useState, useEffect, useRef } from 'react';
import { Outlet, NavLink, useLocation, useNavigate } from 'react-router-dom';
import { useTheme } from '../context/ThemeContext';
import { useUser } from '../context/UserContext';

// Navigation item types
interface NavSubItem {
    path: string;
    label: string;
    icon: string;
}

interface NavItem {
    path?: string;
    label: string;
    icon: string;
    submenu?: NavSubItem[];
}

interface NavSection {
    section: string;
    items: NavItem[];
}

// Notification types with navigation actions
interface Notification {
    id: number;
    type: 'payment' | 'ticket' | 'expiring' | 'message' | 'success';
    icon: string;
    title: string;
    message: string;
    time: string;
    isRead: boolean;
    actionPath: string;
    actionLabel: string;
    relatedId?: string;
}

// Navigation items configuration with submenu support
const navItems: NavSection[] = [
    {
        section: 'Principal',
        items: [
            { path: '/app/dashboard', label: 'Dashboard', icon: '📊' },
        ]
    },
    {
        section: 'Operaciones',
        items: [
            {
                label: 'Servicios',
                icon: '🌐',
                submenu: [
                    { path: '/app/marketing-campaigns', label: 'Agencia de Marketing', icon: '📣' },
                    { path: '/app/web-design/dashboard', label: 'Diseño Web', icon: '🎨' },
                    { path: '/app/domains/dashboard', label: 'Dominios', icon: '🌐' },
                    { path: '/app/hosting/dashboard', label: 'Hosting', icon: '🖥️' },
                    { path: '/app/app-dev/dashboard', label: 'Desarrollo de Aplicaciones', icon: '📱' },
                    { path: '/app/software-dev/dashboard', label: 'Desarrollo de Software (Desktop)', icon: '💻' },
                ]
            },
            { path: '/app/workshop/dashboard', label: 'Taller Técnico', icon: '🔧' },
            { path: '/app/store/dashboard', label: 'Tienda de Equipos', icon: '🏪' },
            { path: '/app/tickets', label: 'Soporte Técnico', icon: '🚨' },
            { path: '/app/billing', label: 'Facturación Online', icon: '💳' },
        ]
    },
    {
        section: 'Gestión',
        items: [
            { path: '/app/clients', label: 'Clientes', icon: '👥' },
            { path: '/app/documents', label: 'Documentos', icon: '📁' },
        ]
    },
    {
        section: 'Sistema',
        items: [
            { path: '/app/settings', label: 'Configuración', icon: '⚙️' },
        ]
    }
];

// Mock notifications with actionable navigation
const mockNotifications: Notification[] = [
    {
        id: 1,
        type: 'payment',
        icon: '💳',
        title: 'Factura #001 pagada',
        message: 'Cliente Acme Corp ha pagado $450.00',
        time: 'Hace 15 min',
        isRead: false,
        actionPath: '/app/billing',
        actionLabel: 'Ver factura',
        relatedId: 'INV-001'
    },
    {
        id: 2,
        type: 'ticket',
        icon: '🎫',
        title: 'Ticket #302 abierto',
        message: 'Beta Technologies reporta problema con hosting',
        time: 'Hace 1h',
        isRead: false,
        actionPath: '/app/tickets',
        actionLabel: 'Atender ticket',
        relatedId: 'TKT-302'
    },
    {
        id: 3,
        type: 'expiring',
        icon: '⚠️',
        title: 'Servicio próximo a vencer',
        message: 'Dominio acme.com vence en 3 días',
        time: 'Hace 2h',
        isRead: false,
        actionPath: '/app/services/domains',
        actionLabel: 'Enviar recordatorio',
        relatedId: 'DOM-acme.com'
    },
];

export default function AppLayout() {
    const { theme, toggleTheme } = useTheme();
    const { user } = useUser();
    const location = useLocation();
    const navigate = useNavigate();
    const [sidebarOpen, setSidebarOpen] = useState(false);

    // States
    const [showNotifications, setShowNotifications] = useState(false);
    const [showUserMenu, setShowUserMenu] = useState(false);
    const [notifications, setNotifications] = useState<Notification[]>(mockNotifications);
    const [openSubmenus, setOpenSubmenus] = useState<string[]>([]);

    // REFS for click outside detection
    const notificationRef = useRef<HTMLDivElement>(null);
    const userMenuRef = useRef<HTMLDivElement>(null);

    // Click Outside Logic - The Clean Way (No invisible overlays!)
    useEffect(() => {
        function handleClickOutside(event: MouseEvent) {
            // Close Notifications if clicked outside
            if (notificationRef.current && !notificationRef.current.contains(event.target as Node)) {
                setShowNotifications(false);
            }
            // Close User Menu if clicked outside
            if (userMenuRef.current && !userMenuRef.current.contains(event.target as Node)) {
                setShowUserMenu(false);
            }
        }
        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    // Auto-open submenu if active
    useEffect(() => {
        navItems.forEach(section => {
            section.items.forEach(item => {
                if (item.submenu && isSubmenuActive(item.submenu)) {
                    if (!openSubmenus.includes(item.label)) {
                        setOpenSubmenus(prev => [...prev, item.label]);
                    }
                }
            });
        });
    }, [location.pathname]);

    const toggleSubmenu = (label: string) => {
        setOpenSubmenus(prev => prev.includes(label) ? prev.filter(l => l !== label) : [...prev, label]);
    };

    const isSubmenuActive = (submenu: NavSubItem[]) => submenu.some(item => location.pathname === item.path);

    const getCurrentTitle = () => {
        for (const section of navItems) {
            for (const item of section.items) {
                if (item.path && location.pathname === item.path) return item.label;
                if (item.submenu) {
                    const subItem = item.submenu.find(sub => location.pathname === sub.path);
                    if (subItem) return subItem.label;
                }
            }
        }
        return 'Dashboard';
    };

    const unreadCount = notifications.filter(n => !n.isRead).length;
    const markAllAsRead = () => setNotifications(prev => prev.map(n => ({ ...n, isRead: true })));

    const handleNotificationClick = (notification: Notification) => {
        setNotifications(prev => prev.map(n => n.id === notification.id ? { ...n, isRead: true } : n));
        setShowNotifications(false);
        navigate(notification.actionPath);
    };

    const handleUserMenuAction = (action: string) => {
        setShowUserMenu(false);
        if (action === 'logout') window.location.href = '/logout';
        if (action === 'settings') navigate('/app/settings');
        if (action === 'profile') navigate('/app/settings');
    };

    const getNotificationBg = (type: string) => {
        switch (type) {
            case 'payment': return 'rgba(16, 185, 129, 0.15)';
            case 'ticket': return 'rgba(239, 68, 68, 0.15)';
            case 'expiring': return 'rgba(245, 158, 11, 0.15)';
            case 'message': return 'rgba(59, 130, 246, 0.15)';
            case 'success': return 'rgba(16, 185, 129, 0.15)';
            default: return 'rgba(100, 116, 139, 0.15)';
        }
    };

    return (
        <div className="app-layout">
            {/* Mobile Overlay for Sidebar only */}
            {sidebarOpen && (
                <div
                    className="mobile-overlay"
                    style={{ position: 'fixed', inset: 0, background: 'rgba(0,0,0,0.5)', zIndex: 35 }}
                    onClick={() => setSidebarOpen(false)}
                />
            )}

            {/* Sidebar */}
            <aside className={`sidebar ${sidebarOpen ? 'open' : ''}`}>
                <div className="sidebar-header">
                    <div className="sidebar-brand">
                        <div className="logo-container"><span className="logo-placeholder">🚀</span></div>
                        <div className="brand-text">
                            <span className="brand-name">Cibertronia</span>
                            <span className="brand-tagline">Service Hub</span>
                        </div>
                    </div>
                </div>
                <nav className="sidebar-nav">
                    {navItems.map((section, sectionIndex) => (
                        <div key={sectionIndex} className="nav-section">
                            <div className="nav-section-title">{section.section}</div>
                            <ul className="nav-list">
                                {section.items.map((item, itemIndex) => (
                                    item.submenu ? (
                                        <li key={item.label} className="nav-item-group">
                                            <button
                                                className={`nav-item-button ${isSubmenuActive(item.submenu) ? 'active' : ''}`}
                                                onClick={() => toggleSubmenu(item.label)}
                                                style={{
                                                    width: '100%', display: 'flex', alignItems: 'center', justifyContent: 'space-between',
                                                    padding: '0.75rem 1rem', border: 'none', borderRadius: '12px', cursor: 'pointer',
                                                    background: isSubmenuActive(item.submenu) ? 'linear-gradient(135deg, rgba(99, 102, 241, 0.15), rgba(139, 92, 246, 0.1))' : 'transparent',
                                                    color: isSubmenuActive(item.submenu) ? 'var(--color-primary)' : 'var(--text-secondary)',
                                                }}
                                            >
                                                <div style={{ display: 'flex', alignItems: 'center', gap: '0.75rem' }}><span className="icon">{item.icon}</span><span>{item.label}</span></div>
                                                <span style={{ fontSize: '0.6rem', transform: openSubmenus.includes(item.label) ? 'rotate(180deg)' : 'rotate(0deg)' }}>▼</span>
                                            </button>
                                            <ul className="submenu" style={{
                                                listStyle: 'none', padding: 0, margin: 0, marginLeft: '1rem',
                                                maxHeight: openSubmenus.includes(item.label) ? `${item.submenu.length * 45}px` : '0',
                                                overflow: 'hidden', transition: 'all 0.3s ease', opacity: openSubmenus.includes(item.label) ? 1 : 0
                                            }}>
                                                {item.submenu.map((subItem) => (
                                                    <li key={subItem.path} className={`nav-item submenu-item ${location.pathname === subItem.path ? 'active' : ''}`}>
                                                        <NavLink to={subItem.path} onClick={() => setSidebarOpen(false)} style={{
                                                            display: 'flex', alignItems: 'center', gap: '0.65rem', padding: '0.6rem 0.85rem',
                                                            borderRadius: '10px', fontSize: '0.85rem', textDecoration: 'none',
                                                            color: location.pathname === subItem.path ? 'var(--color-primary)' : 'var(--text-secondary)',
                                                            background: location.pathname === subItem.path ? 'linear-gradient(135deg, rgba(99, 102, 241, 0.2), rgba(139, 92, 246, 0.15))' : 'transparent',
                                                        }}>
                                                            <span style={{ fontSize: '0.8rem' }}>{subItem.icon}</span><span>{subItem.label}</span>
                                                        </NavLink>
                                                    </li>
                                                ))}
                                            </ul>
                                        </li>
                                    ) : (
                                        <li key={item.path || itemIndex} className={`nav-item ${location.pathname === item.path ? 'active' : ''}`}>
                                            <NavLink to={item.path!} onClick={() => setSidebarOpen(false)}>
                                                <span className="icon">{item.icon}</span><span>{item.label}</span>
                                            </NavLink>
                                        </li>
                                    )
                                ))}
                            </ul>
                        </div>
                    ))}
                </nav>

                {/* Logout Button at Bottom */}
                <div style={{
                    marginTop: 'auto',
                    padding: '1rem',
                    borderTop: '1px solid var(--border-color)'
                }}>
                    <button
                        onClick={() => window.location.href = '/logout'}
                        style={{
                            width: '100%',
                            display: 'flex',
                            alignItems: 'center',
                            gap: '0.75rem',
                            padding: '0.75rem 1rem',
                            border: 'none',
                            borderRadius: '12px',
                            background: 'rgba(239, 68, 68, 0.1)',
                            color: '#ef4444',
                            cursor: 'pointer',
                            fontWeight: 500,
                            transition: 'all 0.2s'
                        }}
                        onMouseEnter={(e) => {
                            e.currentTarget.style.background = 'rgba(239, 68, 68, 0.15)';
                        }}
                        onMouseLeave={(e) => {
                            e.currentTarget.style.background = 'rgba(239, 68, 68, 0.1)';
                        }}
                    >
                        <span className="icon">🚪</span>
                        <span>Cerrar Sesión</span>
                    </button>
                </div>
            </aside>

            {/* Main Content */}
            <div className="main-content">
                <header className="top-bar">
                    <div className="top-bar-left">
                        <button className="mobile-menu-btn" onClick={() => setSidebarOpen(!sidebarOpen)}>☰</button>
                        <div>
                            <div className="top-bar-title">{getCurrentTitle()}</div>
                            <div className="top-bar-breadcrumb">Panel de Administración / <span>{getCurrentTitle()}</span></div>
                        </div>
                    </div>

                    <div className="top-actions">
                        <button className="btn-icon" title="Buscar">🔍</button>
                        <button className="btn-icon" onClick={toggleTheme}> {theme === 'dark' ? '☀️' : '🌙'} </button>

                        {/* Notifications */}
                        <div style={{ position: 'relative' }} ref={notificationRef}>
                            <button
                                className={`btn-icon notification-bell ${unreadCount > 0 ? 'has-unread' : ''}`}
                                onClick={() => { setShowNotifications(!showNotifications); setShowUserMenu(false); }}
                                style={{ cursor: 'pointer' }}
                            >
                                🔔
                                {unreadCount > 0 && <span style={{
                                    position: 'absolute', top: '2px', right: '2px', width: '18px', height: '18px',
                                    background: 'linear-gradient(135deg, #ef4444, #dc2626)', borderRadius: '50%',
                                    fontSize: '0.65rem', fontWeight: 700, color: 'white', display: 'flex', alignItems: 'center', justifyContent: 'center'
                                }}>{unreadCount}</span>}
                            </button>

                            {showNotifications && (
                                <div className="notifications-dropdown" style={{
                                    position: 'absolute', top: 'calc(100% + 10px)', right: 0, width: '380px',
                                    background: theme === 'dark' ? 'rgba(30, 41, 59, 0.95)' : 'rgba(255, 255, 255, 0.95)',
                                    backdropFilter: 'blur(12px)', borderRadius: '16px', boxShadow: '0 20px 50px rgba(0,0,0,0.3)',
                                    border: '1px solid var(--border-color)', zIndex: 1000, overflow: 'hidden'
                                }}>
                                    <div style={{ padding: '1rem 1.25rem', borderBottom: '1px solid var(--border-color)', background: 'var(--bg-secondary)', display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                                        <span style={{ fontWeight: 600 }}>Notificaciones ({unreadCount})</span>
                                        <button className="btn btn-ghost btn-sm" onClick={markAllAsRead} style={{ fontSize: '0.75rem', cursor: 'pointer' }}>✓ Leídas</button>
                                    </div>
                                    <div style={{ maxHeight: '400px', overflowY: 'auto' }}>
                                        {notifications.length === 0 ? (
                                            <div style={{ padding: '2rem', textAlign: 'center', color: 'var(--text-muted)' }}><div>🔔</div><div>No hay notificaciones</div></div>
                                        ) : (
                                            notifications.map(notif => (
                                                <div key={notif.id} onClick={() => handleNotificationClick(notif)} style={{
                                                    padding: '1rem 1.25rem', cursor: 'pointer', borderBottom: '1px solid var(--border-color)',
                                                    background: notif.isRead ? 'transparent' : 'rgba(99, 102, 241, 0.05)', display: 'flex', gap: '0.75rem'
                                                }}>
                                                    <div style={{ width: '40px', height: '40px', borderRadius: '12px', background: getNotificationBg(notif.type), display: 'flex', alignItems: 'center', justifyContent: 'center' }}>{notif.icon}</div>
                                                    <div>
                                                        <div style={{ fontWeight: 600, fontSize: '0.9rem' }}>{notif.title}</div>
                                                        <div style={{ fontSize: '0.8rem', color: 'var(--text-secondary)' }}>{notif.message}</div>
                                                    </div>
                                                </div>
                                            ))
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>

                        {/* User Profile */}
                        <div style={{ position: 'relative' }} ref={userMenuRef}>
                            <div className="profile" onClick={() => {
                                console.log('👤 [AppLayout] Profile clicked! Current showUserMenu:', showUserMenu);
                                setShowUserMenu(!showUserMenu);
                                setShowNotifications(false);
                            }} style={{ cursor: 'pointer' }}>
                                <div className="profile-avatar" style={{
                                    width: '40px', height: '40px', borderRadius: '12px', background: 'linear-gradient(135deg, var(--color-primary), #8b5cf6)',
                                    display: 'flex', alignItems: 'center', justifyContent: 'center', color: 'white', fontWeight: 700
                                }}>
                                    {user?.nombre ? user.nombre.split(' ').map((n: string) => n[0]).join('').toUpperCase().slice(0, 2) : 'JP'}
                                </div>
                                <div className="profile-info">
                                    <span className="profile-name">{user?.nombre || 'Juan Pérez'}</span>
                                    <span style={{ fontSize: '0.6rem' }}>▼</span>
                                </div>
                            </div>

                            {showUserMenu && (() => {
                                console.log('✅ [AppLayout] User menu RENDERING (showUserMenu is TRUE)');
                                return (
                                    <div className="user-menu-dropdown" style={{
                                        position: 'absolute', top: 'calc(100% + 10px)', right: 0, width: '220px',
                                        background: theme === 'dark' ? 'rgba(30, 41, 59, 0.95)' : 'rgba(255, 255, 255, 0.95)',
                                        backdropFilter: 'blur(12px)', borderRadius: '12px', boxShadow: '0 20px 50px rgba(0,0,0,0.3)',
                                        border: '1px solid var(--border-color)', zIndex: 1000, overflow: 'hidden',
                                        pointerEvents: 'auto'
                                    }}>
                                        <div style={{ padding: '1rem', background: 'var(--bg-secondary)', borderBottom: '1px solid var(--border-color)' }}>
<div style={{ fontWeight: 600 }}>{user?.nombre || 'Juan Pérez'}</div>
                                            <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>{user?.email || 'admin@cibertronia.com'}</div>
                                        </div>
                                        <div style={{ padding: '0.5rem' }}>
                                            <button
                                                onClick={(e) => {
                                                    e.stopPropagation();
                                                    console.log('🖱️ [BUTTON CLICK] Profile button clicked!');
                                                    handleUserMenuAction('profile');
                                                }}
                                                onMouseEnter={() => console.log('🖱️ [HOVER] Mouse ENTERED Profile button')}
                                                style={{ width: '100%', padding: '0.75rem', border: 'none', background: 'transparent', display: 'flex', gap: '0.75rem', cursor: 'pointer', color: 'var(--text-primary)' }}
                                            >
                                                <span>👤</span> Mi Perfil
                                            </button>
                                            <button
                                                onClick={(e) => {
                                                    e.stopPropagation();
                                                    console.log('🖱️ [BUTTON CLICK] Settings button clicked!');
                                                    handleUserMenuAction('settings');
                                                }}
                                                onMouseEnter={() => console.log('🖱️ [HOVER] Mouse ENTERED Settings button')}
                                                style={{ width: '100%', padding: '0.75rem', border: 'none', background: 'transparent', display: 'flex', gap: '0.75rem', cursor: 'pointer', color: 'var(--text-primary)' }}
                                            >
                                                <span>⚙️</span> Ajustes
                                            </button>
                                            <div style={{ height: '1px', background: 'var(--border-color)', margin: '0.5rem 0' }} />
                                            <button
                                                onClick={(e) => {
                                                    e.stopPropagation();
                                                    console.log('🚪🚪🚪 [LOGOUT CLICK] Cerrar Sesión button CLICKED!!!');
                                                    handleUserMenuAction('logout');
                                                }}
                                                onMouseEnter={(e) => {
                                                    console.log('🖱️ [HOVER] Mouse ENTERED Logout button');
                                                    e.currentTarget.style.background = 'rgba(239, 68, 68, 0.1)';
                                                }}
                                                onMouseLeave={(e) => {
                                                    console.log('🖱️ [HOVER] Mouse LEFT Logout button');
                                                    e.currentTarget.style.background = 'transparent';
                                                }}
                                                style={{ width: '100%', padding: '0.75rem', border: 'none', background: 'transparent', display: 'flex', gap: '0.75rem', cursor: 'pointer', color: '#ef4444' }}
                                            >
                                                <span>🚪</span> Cerrar Sesión
                                            </button>
                                        </div>
                                    </div>
                                );
                            })()}
                        </div>
                    </div>
                </header>

                <main className="page-content">
                    <Outlet />
                </main>
            </div>
        </div>
    );
}
