import { ReactNode } from 'react';

interface PageHeaderProps {
    title: string;
    subtitle?: string;
    actions?: ReactNode;
    stats?: Array<{
        icon: string;
        label: string;
        value: string | number;
        color?: string;
    }>;
}

/**
 * 🎨 COMPONENTE ESTÁNDAR DE HEADER
 * 
 * Este componente debe usarse en TODAS las páginas para mantener consistencia visual.
 * Proporciona el mismo estilo premium del Dashboard.
 * 
 * Uso:
 * <PageHeader 
 *   title="Título de la Página"
 *   subtitle="Descripción breve"
 *   actions={<button>Acción</button>}
 *   stats={[{ icon: '📊', label: 'Total', value: 42 }]}
 * />
 */
export default function PageHeader({ title, subtitle, actions, stats }: PageHeaderProps) {
    return (
        <>
            {/* Page Header */}
            <div className="page-header">
                <div>
                    <h1 className="page-title">{title}</h1>
                    {subtitle && <p className="page-subtitle">{subtitle}</p>}
                </div>
                {actions && <div style={{ display: 'flex', gap: '0.75rem' }}>{actions}</div>}
            </div>

            {/* Quick Stats Tags (if provided) */}
            {stats && stats.length > 0 && (
                <div className="stats-row" style={{ display: 'flex', gap: '0.75rem', marginBottom: '2rem', flexWrap: 'wrap' }}>
                    {stats.map((stat, index) => (
                        <div key={index} className="stat-tag">
                            <span>{stat.icon}</span>
                            <span>{stat.label}:</span>
                            <span className="stat-tag-value" style={stat.color ? { color: stat.color } : {}}>
                                {stat.value}
                            </span>
                        </div>
                    ))}
                </div>
            )}
        </>
    );
}
