import { useState, useEffect } from 'react';
import axios from 'axios';

interface NewProjectModalProps {
    isOpen: boolean;
    onClose: () => void;
    onSuccess?: () => void;
}

export default function NewProjectModal({ isOpen, onClose, onSuccess }: NewProjectModalProps) {
    const [loading, setLoading] = useState(false);
    const [clients, setClients] = useState<any[]>([]);
    const [plans, setPlans] = useState<any[]>([]);

    const [formData, setFormData] = useState({
        name: '',
        client_id: '',
        plan_id: '',
        platform: 'windows',
        start_date: new Date().toISOString().split('T')[0],
        description: '',
    });

    useEffect(() => {
        if (isOpen) {
            loadData();
        }
    }, [isOpen]);

    const loadData = async () => {
        try {
            const [clientsRes, plansRes] = await Promise.all([
                axios.get('/api/clients'),
                axios.get('/api/software-dev/plans'),
            ]);
            setClients(clientsRes.data.data || clientsRes.data || []);
            setPlans(plansRes.data || []);
        } catch (error) {
            console.error('Error cargando datos:', error);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);

        try {
            await axios.post('/api/software-dev/projects', formData);
            alert('✅ Proyecto creado exitosamente');
            onSuccess?.();
            onClose();
            // Reset form
            setFormData({
                name: '',
                client_id: '',
                plan_id: '',
                platform: 'windows',
                start_date: new Date().toISOString().split('T')[0],
                description: '',
            });
        } catch (error: any) {
            console.error('Error creando proyecto:', error);
            alert('❌ Error: ' + (error.response?.data?.message || 'No se pudo crear el proyecto'));
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div
            style={{
                position: 'fixed',
                inset: 0,
                background: 'rgba(0, 0, 0, 0.7)',
                backdropFilter: 'blur(4px)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                zIndex: 9999,
                padding: '1rem',
            }}
            onClick={onClose}
        >
            <div
                className="card"
                style={{
                    width: '100%',
                    maxWidth: '600px',
                    maxHeight: '90vh',
                    overflowY: 'auto',
                    padding: 0,
                }}
                onClick={(e) => e.stopPropagation()}
            >
                {/* Header */}
                <div style={{
                    padding: '1.5rem',
                    borderBottom: '1px solid var(--border-color)',
                    display: 'flex',
                    justifyContent: 'space-between',
                    alignItems: 'center',
                }}>
                    <div>
                        <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                            💻 Nuevo Proyecto de Software
                        </h2>
                        <p style={{ fontSize: '0.875rem', color: 'var(--text-muted)' }}>
                            Crear nuevo proyecto de aplicación de escritorio
                        </p>
                    </div>
                    <button
                        onClick={onClose}
                        className="btn-icon"
                        style={{ width: '36px', height: '36px' }}
                    >
                        ✕
                    </button>
                </div>

                {/* Form */}
                <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                    <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                        {/* Nombre */}
                        <div>
                            <label className="form-label">
                                Nombre del Proyecto <span style={{ color: 'var(--color-danger)' }}>*</span>
                            </label>
                            <input
                                type="text"
                                className="input"
                                value={formData.name}
                                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                placeholder="Ej: Sistema de Inventario Desktop"
                                required
                            />
                        </div>

                        {/* Cliente y Plan en fila */}
                        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                            <div>
                                <label className="form-label">
                                    Cliente <span style={{ color: 'var(--color-danger)' }}>*</span>
                                </label>
                                <select
                                    className="input"
                                    value={formData.client_id}
                                    onChange={(e) => setFormData({ ...formData, client_id: e.target.value })}
                                    required
                                >
                                    <option value="">Seleccionar cliente</option>
                                    {clients.map((client) => (
                                        <option key={client.id} value={client.id}>
                                            {client.name}
                                        </option>
                                    ))}
                                </select>
                                {clients.length === 0 && (
                                    <p style={{ fontSize: '0.75rem', color: 'var(--color-warning)', marginTop: '0.25rem' }}>
                                        ⚠️ No hay clientes. Crea uno primero.
                                    </p>
                                )}
                            </div>

                            <div>
                                <label className="form-label">Plan de Licenciamiento</label>
                                <select
                                    className="input"
                                    value={formData.plan_id}
                                    onChange={(e) => setFormData({ ...formData, plan_id: e.target.value })}
                                >
                                    <option value="">Sin plan (definir luego)</option>
                                    {plans.map((plan) => (
                                        <option key={plan.id} value={plan.id}>
                                            {plan.name}
                                        </option>
                                    ))}
                                </select>
                            </div>
                        </div>

                        {/* Plataforma y Fecha */}
                        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem' }}>
                            <div>
                                <label className="form-label">
                                    Plataforma <span style={{ color: 'var(--color-danger)' }}>*</span>
                                </label>
                                <select
                                    className="input"
                                    value={formData.platform}
                                    onChange={(e) => setFormData({ ...formData, platform: e.target.value })}
                                    required
                                >
                                    <option value="windows">🪟 Windows</option>
                                    <option value="macos">🍎 macOS</option>
                                    <option value="linux">🐧 Linux</option>
                                    <option value="cross_platform">🌐 Multi-Plataforma</option>
                                </select>
                            </div>

                            <div>
                                <label className="form-label">Fecha de Inicio</label>
                                <input
                                    type="date"
                                    className="input"
                                    value={formData.start_date}
                                    onChange={(e) => setFormData({ ...formData, start_date: e.target.value })}
                                />
                            </div>
                        </div>

                        {/* Descripción */}
                        <div>
                            <label className="form-label">Descripción</label>
                            <textarea
                                className="input"
                                rows={3}
                                value={formData.description}
                                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                                placeholder="Breve descripción del proyecto..."
                                style={{ resize: 'vertical' }}
                            />
                        </div>
                    </div>

                    {/* Actions */}
                    <div style={{ display: 'flex', gap: '0.75rem', marginTop: '2rem', justifyContent: 'flex-end' }}>
                        <button
                            type="button"
                            className="btn btn-secondary"
                            onClick={onClose}
                            disabled={loading}
                        >
                            Cancelar
                        </button>
                        <button
                            type="submit"
                            className="btn btn-primary"
                            disabled={loading || clients.length === 0}
                        >
                            {loading ? '⏳ Creando...' : '✅ Crear Proyecto'}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
}
