import { useState, useEffect } from 'react';
import axios from 'axios';

interface NewOrderModalProps {
    isOpen: boolean;
    onClose: () => void;
    onSuccess: () => void;
}

interface Client {
    id: number;
    name: string;
}

interface Product {
    id: number;
    sku: string;
    name: string;
    price: number;
    stock: number;
    brand?: string;
    model?: string;
}

interface OrderItem {
    product_id: number;
    product_name: string;
    qty: number;
    unit_price: number;
    discount: number;
    subtotal: number;
}

export default function NewOrderModal({ isOpen, onClose, onSuccess }: NewOrderModalProps) {
    const [clients, setClients] = useState<Client[]>([]);
    const [products, setProducts] = useState<Product[]>([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');

    const [formData, setFormData] = useState({
        client_id: '',
        notes: '',
        discount: 0,
    });

    const [items, setItems] = useState<OrderItem[]>([]);
    const [selectedProduct, setSelectedProduct] = useState('');
    const [itemQty, setItemQty] = useState(1);
    const [itemDiscount, setItemDiscount] = useState(0);

    useEffect(() => {
        if (isOpen) {
            fetchClients();
            fetchProducts();
        }
    }, [isOpen]);

    const fetchClients = async () => {
        try {
            const res = await axios.get('/api/clients');
            setClients(res.data.data || res.data);
        } catch (err) {
            console.error('Error fetching clients:', err);
        }
    };

    const fetchProducts = async () => {
        try {
            const res = await axios.get('/api/store/products');
            setProducts(res.data.data || res.data);
        } catch (err) {
            console.error('Error fetching products:', err);
        }
    };

    const addItem = () => {
        if (!selectedProduct) {
            setError('Selecciona un producto');
            return;
        }

        const product = products.find(p => p.id === parseInt(selectedProduct));
        if (!product) return;

        if (itemQty > product.stock) {
            setError(`Stock insuficiente. Disponible: ${product.stock}`);
            return;
        }

        // Verificar si ya existe
        const existingIndex = items.findIndex(i => i.product_id === product.id);
        if (existingIndex >= 0) {
            const newItems = [...items];
            newItems[existingIndex].qty += itemQty;
            newItems[existingIndex].subtotal =
                (newItems[existingIndex].qty * newItems[existingIndex].unit_price) - newItems[existingIndex].discount;
            setItems(newItems);
        } else {
            const subtotal = (itemQty * product.price) - itemDiscount;
            setItems([...items, {
                product_id: product.id,
                product_name: product.name,
                qty: itemQty,
                unit_price: product.price,
                discount: itemDiscount,
                subtotal: subtotal,
            }]);
        }

        // Reset
        setSelectedProduct('');
        setItemQty(1);
        setItemDiscount(0);
        setError('');
    };

    const removeItem = (index: number) => {
        setItems(items.filter((_, i) => i !== index));
    };

    const calculateTotals = () => {
        const subtotal = items.reduce((sum, item) => sum + item.subtotal, 0);
        const total = subtotal - formData.discount;
        return { subtotal, total };
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');

        if (items.length === 0) {
            setError('Agrega al menos un producto a la orden');
            return;
        }

        setLoading(true);

        try {
            const payload = {
                client_id: parseInt(formData.client_id),
                discount: formData.discount,
                notes: formData.notes || null,
                items: items.map(item => ({
                    product_id: item.product_id,
                    qty: item.qty,
                    unit_price: item.unit_price,
                    discount: item.discount,
                })),
            };

            await axios.post('/api/store/orders', payload);

            // Reset
            setFormData({ client_id: '', notes: '', discount: 0 });
            setItems([]);
            onSuccess();
            onClose();
        } catch (err: any) {
            setError(err.response?.data?.message || 'Error al crear la orden');
            console.error('Error creating order:', err);
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    const { subtotal, total } = calculateTotals();

    return (
        <div
            style={{
                position: 'fixed',
                inset: 0,
                background: 'rgba(0, 0, 0, 0.7)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                zIndex: 50,
                padding: '1rem',
            }}
            onClick={onClose}
        >
            <div
                style={{
                    background: 'var(--bg-card)',
                    borderRadius: '16px',
                    maxWidth: '900px',
                    width: '100%',
                    maxHeight: '90vh',
                    overflow: 'auto',
                    boxShadow: '0 20px 25px -5px rgba(0, 0, 0, 0.6)',
                }}
                onClick={(e) => e.stopPropagation()}
            >
                {/* Header */}
                <div style={{
                    padding: '1.5rem',
                    borderBottom: '1px solid var(--border-color)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'space-between',
                    position: 'sticky',
                    top: 0,
                    background: 'var(--bg-card)',
                    zIndex: 1,
                }}>
                    <div>
                        <h2 style={{ fontSize: '1.5rem', fontWeight: 700, marginBottom: '0.25rem' }}>
                            🛒 Nueva Orden de Venta
                        </h2>
                        <p style={{ color: 'var(--text-secondary)', fontSize: '0.875rem' }}>
                            Crea una orden de venta para un cliente
                        </p>
                    </div>
                    <button
                        onClick={onClose}
                        style={{
                            background: 'transparent',
                            border: 'none',
                            fontSize: '1.5rem',
                            cursor: 'pointer',
                            color: 'var(--text-secondary)',
                        }}
                    >
                        ✕
                    </button>
                </div>

                {/* Body */}
                <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                    {error && (
                        <div style={{
                            padding: '1rem',
                            background: 'rgba(239, 68, 68, 0.1)',
                            border: '1px solid rgba(239, 68, 68, 0.3)',
                            borderRadius: '8px',
                            color: '#ef4444',
                            marginBottom: '1rem',
                        }}>
                            {error}
                        </div>
                    )}

                    {/* Cliente */}
                    <div style={{ marginBottom: '1.5rem' }}>
                        <label className="form-label">
                            Cliente <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <select
                            className="input"
                            value={formData.client_id}
                            onChange={(e) => setFormData({ ...formData, client_id: e.target.value })}
                            required
                        >
                            <option value="">Seleccionar cliente...</option>
                            {clients.map((client) => (
                                <option key={client.id} value={client.id}>
                                    {client.name}
                                </option>
                            ))}
                        </select>
                    </div>

                    {/* Agregar Productos */}
                    <div style={{
                        padding: '1.5rem',
                        background: 'var(--glass-bg)',
                        borderRadius: '12px',
                        marginBottom: '1.5rem',
                    }}>
                        <h3 style={{ marginBottom: '1rem', fontSize: '1.1rem' }}>➕ Agregar Producto</h3>
                        <div style={{ display: 'grid', gridTemplateColumns: '2fr 1fr 1fr auto', gap: '0.75rem', alignItems: 'end' }}>
                            <div>
                                <label className="form-label" style={{ fontSize: '0.875rem' }}>Producto</label>
                                <select
                                    className="input"
                                    value={selectedProduct}
                                    onChange={(e) => setSelectedProduct(e.target.value)}
                                >
                                    <option value="">Seleccionar...</option>
                                    {products.filter(p => p.stock > 0).map((product) => (
                                        <option key={product.id} value={product.id}>
                                            {product.name} - ${product.price} (Stock: {product.stock})
                                        </option>
                                    ))}
                                </select>
                            </div>
                            <div>
                                <label className="form-label" style={{ fontSize: '0.875rem' }}>Cantidad</label>
                                <input
                                    type="number"
                                    className="input"
                                    value={itemQty}
                                    onChange={(e) => setItemQty(parseInt(e.target.value) || 1)}
                                    min="1"
                                />
                            </div>
                            <div>
                                <label className="form-label" style={{ fontSize: '0.875rem' }}>Desc. Item</label>
                                <input
                                    type="number"
                                    className="input"
                                    value={itemDiscount}
                                    onChange={(e) => setItemDiscount(parseFloat(e.target.value) || 0)}
                                    min="0"
                                    step="0.01"
                                />
                            </div>
                            <button
                                type="button"
                                className="btn btn-primary"
                                onClick={addItem}
                            >
                                ➕ Agregar
                            </button>
                        </div>
                    </div>

                    {/* Lista de Items */}
                    {items.length > 0 && (
                        <div style={{ marginBottom: '1.5rem' }}>
                            <h3 style={{ marginBottom: '1rem', fontSize: '1.1rem' }}>📦 Items de la Orden</h3>
                            <div className="table-container">
                                <table className="table">
                                    <thead>
                                        <tr>
                                            <th>Producto</th>
                                            <th>Cant.</th>
                                            <th>Precio Unit.</th>
                                            <th>Desc.</th>
                                            <th>Subtotal</th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {items.map((item, index) => (
                                            <tr key={index}>
                                                <td>{item.product_name}</td>
                                                <td>{item.qty}</td>
                                                <td>${item.unit_price.toFixed(2)}</td>
                                                <td>
                                                    {item.discount > 0 ? (
                                                        <span style={{ color: '#10b981' }}>-${item.discount.toFixed(2)}</span>
                                                    ) : '—'}
                                                </td>
                                                <td><strong>${item.subtotal.toFixed(2)}</strong></td>
                                                <td>
                                                    <button
                                                        type="button"
                                                        onClick={() => removeItem(index)}
                                                        style={{
                                                            background: 'transparent',
                                                            border: 'none',
                                                            color: '#ef4444',
                                                            cursor: 'pointer',
                                                            fontSize: '1.2rem',
                                                        }}
                                                    >
                                                        🗑️
                                                    </button>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    )}

                    {/* Descuento Global y Notas */}
                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem', marginBottom: '1.5rem' }}>
                        <div>
                            <label className="form-label">Descuento Global</label>
                            <input
                                type="number"
                                className="input"
                                value={formData.discount}
                                onChange={(e) => setFormData({ ...formData, discount: parseFloat(e.target.value) || 0 })}
                                min="0"
                                step="0.01"
                                placeholder="0.00"
                            />
                        </div>
                        <div>
                            <label className="form-label">Notas</label>
                            <input
                                type="text"
                                className="input"
                                value={formData.notes}
                                onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                                placeholder="Observaciones..."
                            />
                        </div>
                    </div>

                    {/* Totales */}
                    {items.length > 0 && (
                        <div style={{
                            padding: '1.5rem',
                            background: 'var(--glass-bg)',
                            borderRadius: '12px',
                            marginBottom: '1.5rem',
                        }}>
                            <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '0.5rem' }}>
                                <span style={{ color: 'var(--text-secondary)' }}>Subtotal:</span>
                                <span style={{ fontWeight: 600 }}>${subtotal.toFixed(2)}</span>
                            </div>
                            {formData.discount > 0 && (
                                <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '0.5rem' }}>
                                    <span style={{ color: 'var(--text-secondary)' }}>Descuento Global:</span>
                                    <span style={{ fontWeight: 600, color: '#10b981' }}>-${formData.discount.toFixed(2)}</span>
                                </div>
                            )}
                            <hr style={{ margin: '1rem 0', border: 'none', borderTop: '1px solid var(--border-color)' }} />
                            <div style={{ display: 'flex', justifyContent: 'space-between', fontSize: '1.25rem' }}>
                                <span style={{ fontWeight: 700 }}>Total:</span>
                                <span style={{ fontWeight: 700, color: 'var(--color-primary)' }}>
                                    ${total.toFixed(2)}
                                </span>
                            </div>
                        </div>
                    )}

                    {/* Footer */}
                    <div style={{
                        display: 'flex',
                        gap: '1rem',
                        paddingTop: '1rem',
                        borderTop: '1px solid var(--border-color)',
                    }}>
                        <button
                            type="button"
                            onClick={onClose}
                            className="btn btn-ghost"
                            style={{ flex: 1 }}
                        >
                            Cancelar
                        </button>
                        <button
                            type="submit"
                            className="btn btn-primary"
                            disabled={loading || items.length === 0}
                            style={{ flex: 1 }}
                        >
                            {loading ? 'Creando...' : '✅ Crear Orden'}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
}
