import { useState, useEffect } from 'react';
import axios from 'axios';

interface NewAssetModalProps {
    isOpen: boolean;
    onClose: () => void;
    onSuccess: () => void;
    preselectedClientId?: number;
}

interface Client {
    id: number;
    name: string;
}

export default function NewAssetModal({ isOpen, onClose, onSuccess, preselectedClientId }: NewAssetModalProps) {
    const [clients, setClients] = useState<Client[]>([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState('');

    const [formData, setFormData] = useState({
        client_id: preselectedClientId?.toString() || '',
        type: 'laptop',
        brand: '',
        model: '',
        serial_number: '',
        purchase_date: '',
        notes: '',
    });

    useEffect(() => {
        if (isOpen) {
            fetchClients();
        }
    }, [isOpen]);

    useEffect(() => {
        if (preselectedClientId) {
            setFormData(prev => ({ ...prev, client_id: preselectedClientId.toString() }));
        }
    }, [preselectedClientId]);

    const fetchClients = async () => {
        try {
            const res = await axios.get('/api/clients');
            setClients(res.data.data || res.data);
        } catch (err) {
            console.error('Error fetching clients:', err);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);

        try {
            const payload: any = {
                client_id: parseInt(formData.client_id),
                type: formData.type,
            };

            if (formData.brand) payload.brand = formData.brand;
            if (formData.model) payload.model = formData.model;
            if (formData.serial_number) payload.serial_number = formData.serial_number;
            if (formData.purchase_date) payload.purchase_date = formData.purchase_date;
            if (formData.notes) payload.notes = formData.notes;

            await axios.post('/api/workshop/assets', payload);

            // Reset form
            setFormData({
                client_id: '',
                type: 'laptop',
                brand: '',
                model: '',
                serial_number: '',
                purchase_date: '',
                notes: '',
            });

            onSuccess();
            onClose();
        } catch (err: any) {
            setError(err.response?.data?.message || 'Error al registrar el equipo');
            console.error('Error creating asset:', err);
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div
            style={{
                position: 'fixed',
                inset: 0,
                background: 'rgba(0, 0, 0, 0.7)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                zIndex: 50,
                padding: '1rem',
            }}
            onClick={onClose}
        >
            <div
                style={{
                    background: 'var(--bg-card)',
                    borderRadius: '16px',
                    maxWidth: '550px',
                    width: '100%',
                    maxHeight: '90vh',
                    overflow: 'auto',
                    boxShadow: '0 20px 25px -5px rgba(0, 0, 0, 0.6)',
                }}
                onClick={(e) => e.stopPropagation()}
            >
                {/* Header */}
                <div style={{
                    padding: '1.5rem',
                    borderBottom: '1px solid var(--border-color)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'space-between',
                }}>
                    <div>
                        <h2 style={{ fontSize: '1.5rem', fontWeight: 700, marginBottom: '0.25rem' }}>
                            📦 Registrar Equipo
                        </h2>
                        <p style={{ color: 'var(--text-secondary)', fontSize: '0.875rem' }}>
                            Añade un equipo al inventario del cliente
                        </p>
                    </div>
                    <button
                        onClick={onClose}
                        style={{
                            background: 'transparent',
                            border: 'none',
                            fontSize: '1.5rem',
                            cursor: 'pointer',
                            color: 'var(--text-secondary)',
                        }}
                    >
                        ✕
                    </button>
                </div>

                {/* Body */}
                <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                    {error && (
                        <div style={{
                            padding: '1rem',
                            background: 'rgba(239, 68, 68, 0.1)',
                            border: '1px solid rgba(239, 68, 68, 0.3)',
                            borderRadius: '8px',
                            color: '#ef4444',
                            marginBottom: '1rem',
                        }}>
                            {error}
                        </div>
                    )}

                    {/* Cliente */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Cliente <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <select
                            className="input"
                            value={formData.client_id}
                            onChange={(e) => setFormData({ ...formData, client_id: e.target.value })}
                            required
                        >
                            <option value="">Seleccionar cliente...</option>
                            {clients.map((client) => (
                                <option key={client.id} value={client.id}>
                                    {client.name}
                                </option>
                            ))}
                        </select>
                    </div>

                    {/* Tipo */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Tipo de Equipo <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <select
                            className="input"
                            value={formData.type}
                            onChange={(e) => setFormData({ ...formData, type: e.target.value })}
                            required
                        >
                            <option value="laptop">💻 Laptop</option>
                            <option value="desktop">🖥️ Desktop</option>
                            <option value="printer">🖨️ Impresora</option>
                            <option value="router">📡 Router</option>
                            <option value="ups">🔋 UPS</option>
                            <option value="server">🖲️ Servidor</option>
                            <option value="monitor">🖥️ Monitor</option>
                            <option value="other">📦 Otro</option>
                        </select>
                    </div>

                    {/* Marca y Modelo */}
                    <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '1rem', marginBottom: '1.25rem' }}>
                        <div>
                            <label className="form-label">Marca</label>
                            <input
                                type="text"
                                className="input"
                                value={formData.brand}
                                onChange={(e) => setFormData({ ...formData, brand: e.target.value })}
                                placeholder="Ej: HP, Dell, Lenovo"
                            />
                        </div>
                        <div>
                            <label className="form-label">Modelo</label>
                            <input
                                type="text"
                                className="input"
                                value={formData.model}
                                onChange={(e) => setFormData({ ...formData, model: e.target.value })}
                                placeholder="Ej: Pavilion 15"
                            />
                        </div>
                    </div>

                    {/* Serial */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">Número de Serie</label>
                        <input
                            type="text"
                            className="input"
                            value={formData.serial_number}
                            onChange={(e) => setFormData({ ...formData, serial_number: e.target.value })}
                            placeholder="Ej: SN123456789"
                        />
                    </div>

                    {/* Fecha de Compra */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">Fecha de Compra</label>
                        <input
                            type="date"
                            className="input"
                            value={formData.purchase_date}
                            onChange={(e) => setFormData({ ...formData, purchase_date: e.target.value })}
                        />
                    </div>

                    {/* Notas */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">Notas</label>
                        <textarea
                            className="input"
                            value={formData.notes}
                            onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                            placeholder="Observaciones adicionales..."
                            rows={2}
                        />
                    </div>

                    {/* Footer */}
                    <div style={{
                        display: 'flex',
                        gap: '1rem',
                        marginTop: '2rem',
                        paddingTop: '1rem',
                        borderTop: '1px solid var(--border-color)',
                    }}>
                        <button
                            type="button"
                            onClick={onClose}
                            className="btn btn-ghost"
                            style={{ flex: 1 }}
                        >
                            Cancelar
                        </button>
                        <button
                            type="submit"
                            className="btn btn-primary"
                            disabled={loading}
                            style={{ flex: 1 }}
                        >
                            {loading ? 'Registrando...' : '✅ Registrar Equipo'}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
}
