import { useState, useEffect } from 'react';
import axios from 'axios';

interface NewWorkOrderModalProps {
    isOpen: boolean;
    onClose: () => void;
    onSuccess: () => void;
}

interface Client {
    id: number;
    name: string;
}

interface Asset {
    id: number;
    asset_tag: string;
    type: string;
    brand?: string;
    model?: string;
}

export default function NewWorkOrderModal({ isOpen, onClose, onSuccess }: NewWorkOrderModalProps) {
    const [clients, setClients] = useState<Client[]>([]);
    const [assets, setAssets] = useState<Asset[]>([]);
    const [loading, setLoading] = useState(false);
    const [loadingData, setLoadingData] = useState(false);
    const [error, setError] = useState('');

    const [formData, setFormData] = useState({
        client_id: '',
        asset_id: '',
        issue_description: '',
        intake_notes: '',
        priority: 'medium',
        estimated_cost: '',
    });

    useEffect(() => {
        if (isOpen) {
            fetchClients();
        }
    }, [isOpen]);

    useEffect(() => {
        if (formData.client_id) {
            fetchClientAssets(parseInt(formData.client_id));
        } else {
            setAssets([]);
        }
    }, [formData.client_id]);

    const fetchClients = async () => {
        try {
            setLoadingData(true);
            const res = await axios.get('/api/clients');
            setClients(res.data.data || res.data);
        } catch (err) {
            console.error('Error fetching clients:', err);
        } finally {
            setLoadingData(false);
        }
    };

    const fetchClientAssets = async (clientId: number) => {
        try {
            const res = await axios.get(`/api/workshop/assets?client_id=${clientId}`);
            setAssets(res.data.data || res.data);
        } catch (err) {
            console.error('Error fetching assets:', err);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setLoading(true);

        try {
            const payload: any = {
                client_id: parseInt(formData.client_id),
                issue_description: formData.issue_description,
                priority: formData.priority,
            };

            if (formData.asset_id) {
                payload.asset_id = parseInt(formData.asset_id);
            }

            if (formData.intake_notes) {
                payload.intake_notes = formData.intake_notes;
            }

            if (formData.estimated_cost) {
                payload.estimated_cost = parseFloat(formData.estimated_cost);
            }

            await axios.post('/api/workshop/work-orders', payload);

            // Reset form
            setFormData({
                client_id: '',
                asset_id: '',
                issue_description: '',
                intake_notes: '',
                priority: 'medium',
                estimated_cost: '',
            });

            onSuccess();
            onClose();
        } catch (err: any) {
            setError(err.response?.data?.message || 'Error al crear la orden de trabajo');
            console.error('Error creating work order:', err);
        } finally {
            setLoading(false);
        }
    };

    if (!isOpen) return null;

    return (
        <div
            style={{
                position: 'fixed',
                inset: 0,
                background: 'rgba(0, 0, 0, 0.7)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                zIndex: 50,
                padding: '1rem',
            }}
            onClick={onClose}
        >
            <div
                style={{
                    background: 'var(--bg-card)',
                    borderRadius: '16px',
                    maxWidth: '600px',
                    width: '100%',
                    maxHeight: '90vh',
                    overflow: 'auto',
                    boxShadow: '0 20px 25px -5px rgba(0, 0, 0, 0.6)',
                }}
                onClick={(e) => e.stopPropagation()}
            >
                {/* Header */}
                <div style={{
                    padding: '1.5rem',
                    borderBottom: '1px solid var(--border-color)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'space-between',
                }}>
                    <div>
                        <h2 style={{ fontSize: '1.5rem', fontWeight: 700, marginBottom: '0.25rem' }}>
                            🔧 Nueva Orden de Trabajo
                        </h2>
                        <p style={{ color: 'var(--text-secondary)', fontSize: '0.875rem' }}>
                            Registra un equipo para reparación o mantenimiento
                        </p>
                    </div>
                    <button
                        onClick={onClose}
                        style={{
                            background: 'transparent',
                            border: 'none',
                            fontSize: '1.5rem',
                            cursor: 'pointer',
                            color: 'var(--text-secondary)',
                        }}
                    >
                        ✕
                    </button>
                </div>

                {/* Body */}
                <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                    {error && (
                        <div style={{
                            padding: '1rem',
                            background: 'rgba(239, 68, 68, 0.1)',
                            border: '1px solid rgba(239, 68, 68, 0.3)',
                            borderRadius: '8px',
                            color: '#ef4444',
                            marginBottom: '1rem',
                        }}>
                            {error}
                        </div>
                    )}

                    {/* Cliente */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Cliente <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <select
                            className="input"
                            value={formData.client_id}
                            onChange={(e) => setFormData({ ...formData, client_id: e.target.value, asset_id: '' })}
                            required
                            disabled={loadingData}
                        >
                            <option value="">Seleccionar cliente...</option>
                            {clients.map((client) => (
                                <option key={client.id} value={client.id}>
                                    {client.name}
                                </option>
                            ))}
                        </select>
                    </div>

                    {/* Equipo (Opcional) */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Equipo Registrado (opcional)
                        </label>
                        <select
                            className="input"
                            value={formData.asset_id}
                            onChange={(e) => setFormData({ ...formData, asset_id: e.target.value })}
                            disabled={!formData.client_id || assets.length === 0}
                        >
                            <option value="">Sin equipo registrado</option>
                            {assets.map((asset) => (
                                <option key={asset.id} value={asset.id}>
                                    {asset.asset_tag} - {asset.brand} {asset.model}
                                </option>
                            ))}
                        </select>
                        {formData.client_id && assets.length === 0 && (
                            <div style={{ fontSize: '0.875rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Este cliente no tiene equipos registrados
                            </div>
                        )}
                    </div>

                    {/* Descripción del Problema */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Descripción del Problema <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <textarea
                            className="input"
                            value={formData.issue_description}
                            onChange={(e) => setFormData({ ...formData, issue_description: e.target.value })}
                            placeholder="Ej: Equipo no enciende, requiere diagnóstico..."
                            rows={3}
                            required
                        />
                    </div>

                    {/* Notas de Recepción */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Notas de Recepción
                        </label>
                        <textarea
                            className="input"
                            value={formData.intake_notes}
                            onChange={(e) => setFormData({ ...formData, intake_notes: e.target.value })}
                            placeholder="Observaciones al recibir el equipo..."
                            rows={2}
                        />
                    </div>

                    {/* Prioridad */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Prioridad <span style={{ color: '#ef4444' }}>*</span>
                        </label>
                        <select
                            className="input"
                            value={formData.priority}
                            onChange={(e) => setFormData({ ...formData, priority: e.target.value })}
                            required
                        >
                            <option value="low">🔵 Baja</option>
                            <option value="medium">🟡 Media</option>
                            <option value="high">🟠 Alta</option>
                            <option value="urgent">🔴 Urgente</option>
                        </select>
                    </div>

                    {/* Costo Estimado */}
                    <div style={{ marginBottom: '1.25rem' }}>
                        <label className="form-label">
                            Costo Estimado (opcional)
                        </label>
                        <input
                            type="number"
                            className="input"
                            value={formData.estimated_cost}
                            onChange={(e) => setFormData({ ...formData, estimated_cost: e.target.value })}
                            placeholder="0.00"
                            step="0.01"
                            min="0"
                        />
                    </div>

                    {/* Footer */}
                    <div style={{
                        display: 'flex',
                        gap: '1rem',
                        marginTop: '2rem',
                        paddingTop: '1rem',
                        borderTop: '1px solid var(--border-color)',
                    }}>
                        <button
                            type="button"
                            onClick={onClose}
                            className="btn btn-ghost"
                            style={{ flex: 1 }}
                        >
                            Cancelar
                        </button>
                        <button
                            type="submit"
                            className="btn btn-primary"
                            disabled={loading}
                            style={{ flex: 1 }}
                        >
                            {loading ? 'Creando...' : '✅ Crear Orden'}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
}
