import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Cie10 } from '../../../types/HistoriaClinica';

interface Cie10SelectProps {
    onSelect: (diagnostico: Cie10) => void;
    selectedId?: number | null;
}

export default function Cie10Select({ onSelect, selectedId }: Cie10SelectProps) {
    const [query, setQuery] = useState('');
    const [results, setResults] = useState<Cie10[]>([]);
    const [loading, setLoading] = useState(false);
    const [showDropdown, setShowDropdown] = useState(false);
    const [selectedItem, setSelectedItem] = useState<Cie10 | null>(null);

    // Initial load / Search
    useEffect(() => {
        const fetchCie10 = async () => {
            setLoading(true);
            try {
                // If query is empty, it fetches popular diagnoses
                const response = await axios.get(`/cie10/buscar?q=${query}`);
                setResults(response.data);
            } catch (error) {
                console.error("Error fetching CIE-10:", error);
            } finally {
                setLoading(false);
            }
        };

        const timer = setTimeout(() => {
            fetchCie10();
        }, 300); // Debounce

        return () => clearTimeout(timer);
    }, [query]);

    const handleSelect = (item: Cie10) => {
        setSelectedItem(item);
        onSelect(item);
        setShowDropdown(false);
        setQuery(''); // Clear search or keep it? Keeping selected item name is better UX usually, handled below.
    };

    return (
        <div className="relative w-full">
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                Diagnóstico CIE-10 (Principal)
            </label>

            {/* Selected Item Display */}
            {selectedItem ? (
                <div className="flex items-center justify-between p-3 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg mb-2">
                    <div>
                        <span className="font-bold text-blue-800 dark:text-blue-300">{selectedItem.codigo}</span>
                        <span className="ml-2 text-blue-900 dark:text-blue-100">{selectedItem.descripcion}</span>
                    </div>
                    <button
                        onClick={() => setSelectedItem(null)}
                        className="text-red-500 hover:text-red-700 font-bold px-2"
                        type="button"
                    >
                        ✕
                    </button>
                </div>
            ) : (
                <div className="relative">
                    <input
                        type="text"
                        className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 transition-all"
                        placeholder="Buscar por código (E11) o nombre (Diabetes)..."
                        value={query}
                        onChange={(e) => {
                            setQuery(e.target.value);
                            setShowDropdown(true);
                        }}
                        onFocus={() => setShowDropdown(true)}
                    />
                    {loading && (
                        <div className="absolute right-3 top-3">
                            <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                        </div>
                    )}
                </div>
            )}

            {/* Dropdown Results */}
            {showDropdown && !selectedItem && (
                <div className="absolute z-50 w-full mt-1 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg shadow-xl max-h-60 overflow-y-auto">
                    {results.length > 0 ? (
                        results.map((item) => (
                            <div
                                key={item.id}
                                onClick={() => handleSelect(item)}
                                className="p-3 hover:bg-blue-50 dark:hover:bg-gray-700 cursor-pointer border-b border-gray-100 dark:border-gray-700 last:border-0 transition-colors"
                            >
                                <div className="flex justify-between items-start">
                                    <span className="font-bold text-blue-600 dark:text-blue-400 min-w-[60px]">{item.codigo}</span>
                                    <span className="text-gray-700 dark:text-gray-200 flex-1">{item.descripcion}</span>
                                </div>
                                {item.categoria && (
                                    <div className="text-xs text-gray-400 mt-1 ml-[60px]">{item.categoria}</div>
                                )}
                            </div>
                        ))
                    ) : (
                        <div className="p-4 text-center text-gray-500 dark:text-gray-400">
                            No se encontraron diagnósticos.
                        </div>
                    )}
                </div>
            )}

            {/* Overlay to close dropdown when clicking outside */}
            {showDropdown && (
                <div
                    className="fixed inset-0 z-40"
                    onClick={() => setShowDropdown(false)}
                />
            )}
        </div>
    );
}
