import React, { useState } from 'react';

export interface MedicamentoItem {
    nombre_medicamento: string;
    dosis: string;
    frecuencia: string;
    duracion_dias: number;
    cantidad_total: number;
    instrucciones_especiales?: string;
}

interface RecetaFormProps {
    medicamentos: MedicamentoItem[];
    onChangeMedicamentos: (meds: MedicamentoItem[]) => void;
    instrucciones: string;
    onChangeInstrucciones: (instr: string) => void;
}

export default function RecetaForm({ medicamentos, onChangeMedicamentos, instrucciones, onChangeInstrucciones }: RecetaFormProps) {
    const handleAdd = () => {
        onChangeMedicamentos([...medicamentos, {
            nombre_medicamento: '',
            dosis: '',
            frecuencia: '',
            duracion_dias: 3,
            cantidad_total: 1
        }]);
    };

    const handleChange = (index: number, field: keyof MedicamentoItem, value: any) => {
        const newMeds = [...medicamentos];
        newMeds[index] = { ...newMeds[index], [field]: value };
        onChangeMedicamentos(newMeds);
    };

    const handleRemove = (index: number) => {
        onChangeMedicamentos(medicamentos.filter((_, i) => i !== index));
    };

    return (
        <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                💊 Receta Médica
            </h3>

            {/* General Instructions */}
            <div className="mb-6">
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                    Instrucciones Generales (Visible en cabecera de receta)
                </label>
                <textarea
                    className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 h-20"
                    placeholder="Ej: Reposo relativo por 3 días, dieta blanda..."
                    value={instrucciones}
                    onChange={(e) => onChangeInstrucciones(e.target.value)}
                />
            </div>

            {/* Medications List */}
            <div className="space-y-4">
                {medicamentos.map((med, index) => (
                    <div key={index} className="p-4 bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600 relative">
                        <button
                            type="button"
                            onClick={() => handleRemove(index)}
                            className="absolute top-2 right-2 text-red-500 hover:text-red-700"
                            title="Eliminar medicamento"
                        >
                            ✕
                        </button>

                        <div className="grid grid-cols-1 md:grid-cols-12 gap-4">
                            <div className="md:col-span-4">
                                <label className="block text-xs font-medium mb-1">Medicamento</label>
                                <input
                                    type="text"
                                    className="w-full p-2 border rounded text-sm"
                                    placeholder="Nombre comercial / genérico"
                                    value={med.nombre_medicamento}
                                    onChange={(e) => handleChange(index, 'nombre_medicamento', e.target.value)}
                                />
                            </div>
                            <div className="md:col-span-2">
                                <label className="block text-xs font-medium mb-1">Dosis</label>
                                <input
                                    type="text"
                                    className="w-full p-2 border rounded text-sm"
                                    placeholder="500mg"
                                    value={med.dosis}
                                    onChange={(e) => handleChange(index, 'dosis', e.target.value)}
                                />
                            </div>
                            <div className="md:col-span-3">
                                <label className="block text-xs font-medium mb-1">Frecuencia</label>
                                <input
                                    type="text"
                                    className="w-full p-2 border rounded text-sm"
                                    placeholder="Cada 8 horas"
                                    value={med.frecuencia}
                                    onChange={(e) => handleChange(index, 'frecuencia', e.target.value)}
                                />
                            </div>
                            <div className="md:col-span-1">
                                <label className="block text-xs font-medium mb-1">Días</label>
                                <input
                                    type="number"
                                    className="w-full p-2 border rounded text-sm"
                                    value={med.duracion_dias}
                                    onChange={(e) => handleChange(index, 'duracion_dias', parseInt(e.target.value))}
                                />
                            </div>
                            <div className="md:col-span-2">
                                <label className="block text-xs font-medium mb-1">Total</label>
                                <input
                                    type="number"
                                    className="w-full p-2 border rounded text-sm"
                                    value={med.cantidad_total}
                                    onChange={(e) => handleChange(index, 'cantidad_total', parseInt(e.target.value))}
                                />
                            </div>
                        </div>
                    </div>
                ))}

                <button
                    type="button"
                    onClick={handleAdd}
                    className="w-full py-3 border-2 border-dashed border-blue-300 text-blue-600 rounded-lg hover:bg-blue-50 font-medium transition-colors"
                >
                    + Agregar Medicamento
                </button>
            </div>
        </div>
    );
}
