import React, { useState, useEffect } from 'react';

interface SignosVitalesProps {
    signos: {
        presion_arterial: string;
        frecuencia_cardiaca: string;
        frecuencia_respiratoria: string;
        temperatura: string;
        saturacion_oxigeno: string;
        peso: string;
        altura: string;
    };
    onChange: (signos: any) => void;
}

export default function SignosVitalesForm({ signos, onChange }: SignosVitalesProps) {
    const [imc, setImc] = useState<number | null>(null);
    const [imcClass, setImcClass] = useState<string>('');

    useEffect(() => {
        if (signos.peso && signos.altura) {
            const peso = parseFloat(signos.peso);
            const alturaCm = parseFloat(signos.altura);
            if (peso > 0 && alturaCm > 0) {
                const alturaM = alturaCm / 100;
                const calcImc = peso / (alturaM * alturaM);
                setImc(calcImc);

                if (calcImc < 18.5) setImcClass('bg-blue-100 text-blue-800');
                else if (calcImc < 25) setImcClass('bg-green-100 text-green-800');
                else if (calcImc < 30) setImcClass('bg-yellow-100 text-yellow-800');
                else setImcClass('bg-red-100 text-red-800');
            } else {
                setImc(null);
            }
        } else {
            setImc(null);
        }
    }, [signos.peso, signos.altura]);

    const handleChange = (field: string, value: string) => {
        onChange({ ...signos, [field]: value });
    };

    return (
        <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                💓 Sección 4: Signos Vitales y Antropometría
            </h3>

            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Presión Arterial</label>
                    <input
                        type="text"
                        placeholder="120/80"
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500"
                        value={signos.presion_arterial}
                        onChange={(e) => handleChange('presion_arterial', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Frec. Cardiaca (lpm)</label>
                    <input
                        type="number"
                        placeholder="60 - 100"
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500"
                        value={signos.frecuencia_cardiaca}
                        onChange={(e) => handleChange('frecuencia_cardiaca', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Frec. Respiratoria</label>
                    <input
                        type="number"
                        placeholder="12 - 20"
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500"
                        value={signos.frecuencia_respiratoria}
                        onChange={(e) => handleChange('frecuencia_respiratoria', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Temperatura (°C)</label>
                    <input
                        type="number"
                        step="0.1"
                        placeholder="36.5"
                        className={`w-full p-2 border rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500 ${parseFloat(signos.temperatura) > 37.5 ? 'border-red-500 bg-red-50 text-red-900' : 'border-gray-300 dark:border-gray-600'}`}
                        value={signos.temperatura}
                        onChange={(e) => handleChange('temperatura', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Saturación O2 (%)</label>
                    <input
                        type="number"
                        placeholder="95 - 100"
                        className={`w-full p-2 border rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500 ${parseFloat(signos.saturacion_oxigeno) < 90 ? 'border-red-500 bg-red-50 text-red-900' : 'border-gray-300 dark:border-gray-600'}`}
                        value={signos.saturacion_oxigeno}
                        onChange={(e) => handleChange('saturacion_oxigeno', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Peso (kg)</label>
                    <input
                        type="number"
                        step="0.1"
                        placeholder="70.5"
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500"
                        value={signos.peso}
                        onChange={(e) => handleChange('peso', e.target.value)}
                    />
                </div>
                <div>
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Altura (cm)</label>
                    <input
                        type="number"
                        placeholder="170"
                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-lg dark:bg-gray-700 focus:ring-2 focus:ring-blue-500"
                        value={signos.altura}
                        onChange={(e) => handleChange('altura', e.target.value)}
                    />
                </div>

                {/* Visualizador de IMC */}
                <div className="flex flex-col justify-center">
                    <label className="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Índice Masa Corporal</label>
                    {imc ? (
                        <div className={`px-3 py-2 rounded-lg font-bold text-center border ${imcClass}`}>
                            {imc.toFixed(2)}
                        </div>
                    ) : (
                        <div className="px-3 py-2 rounded-lg bg-gray-100 dark:bg-gray-700 text-gray-400 text-center text-sm border border-gray-200 dark:border-gray-600">
                            Faltan datos
                        </div>
                    )}
                </div>
            </div>
            {(parseFloat(signos.temperatura) > 37.5 || parseFloat(signos.saturacion_oxigeno) < 90) && (
                <div className="mt-4 p-3 bg-red-50 border-l-4 border-red-500 text-red-700 text-sm font-bold rounded-r-lg">
                    ⚠️ Alerta: Existen signos vitales fuera de rango normal.
                </div>
            )}
        </div>
    );
}
