import React, { useState } from 'react';
import { SoapNotes, ExamenFisico } from '../../../types/HistoriaClinica';

interface SoapFormProps {
    soap: SoapNotes;
    onChangeSoap: (soap: SoapNotes) => void;
    examenFisico: ExamenFisico[];
    onChangeExamen: (examen: ExamenFisico[]) => void;
}

export default function SoapForm({ soap, onChangeSoap, examenFisico, onChangeExamen }: SoapFormProps) {
    const [activeTab, setActiveTab] = useState<'objetivo' | 'analisis' | 'plan'>('objetivo');
    const handleChange = (field: keyof SoapNotes, value: string) => {
        onChangeSoap({ ...soap, [field]: value });
    };

    const handleAddExam = () => {
        onChangeExamen([...examenFisico, { sistema: '', hallazgos: '' }]);
    };

    const handleExamChange = (index: number, field: keyof ExamenFisico, value: string) => {
        const newExamen = [...examenFisico];
        newExamen[index] = { ...newExamen[index], [field]: value };
        onChangeExamen(newExamen);
    };

    const handleRemoveExam = (index: number) => {
        const newExamen = examenFisico.filter((_, i) => i !== index);
        onChangeExamen(newExamen);
    };

    // Predefined systems for ease of use
    const sistemasComunes = [
        'Cardiovascular', 'Respiratorio', 'Abdomen', 'Neurológico',
        'Piel y Faneras', 'Cabeza y Cuello', 'Genitourinario'
    ];

    return (
        <div className="space-y-6">
            {/* SOAP Section */}
            <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                    📝 Sección 5: O.A.P. (Objetivo, Análisis, Plan)
                </h3>

                {/* Tabs Header */}
                <div className="flex border-b border-gray-200 dark:border-gray-700 mb-4">
                    <button
                        className={`py-2 px-4 border-b-2 font-medium text-sm ${activeTab === 'objetivo' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'}`}
                        onClick={() => setActiveTab('objetivo')}
                    >
                        (O) Objetivo
                    </button>
                    <button
                        className={`py-2 px-4 border-b-2 font-medium text-sm ${activeTab === 'analisis' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'}`}
                        onClick={() => setActiveTab('analisis')}
                    >
                        (A) Análisis
                    </button>
                    <button
                        className={`py-2 px-4 border-b-2 font-medium text-sm ${activeTab === 'plan' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'}`}
                        onClick={() => setActiveTab('plan')}
                    >
                        (P) Plan
                    </button>
                </div>

                {/* Tabs Content */}
                <div>
                    {activeTab === 'objetivo' && (
                        <div className="animate-fadeIn">
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                <span className="text-xs text-gray-500">Lo que el médico observa (examen físico general, etc)</span>
                            </label>
                            <textarea
                                className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 min-h-[150px]"
                                placeholder="Signos observados, hallazgos relevantes..."
                                value={soap.objetivo || ''}
                                onChange={(e) => handleChange('objetivo', e.target.value)}
                            />
                        </div>
                    )}

                    {activeTab === 'analisis' && (
                        <div className="animate-fadeIn">
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                <span className="text-xs text-gray-500">Diagnóstico y razonamiento clínico</span>
                            </label>
                            <textarea
                                className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 min-h-[150px]"
                                placeholder="Interpretación clínica, diagnósticos diferenciales..."
                                value={soap.analisis || ''}
                                onChange={(e) => handleChange('analisis', e.target.value)}
                            />
                        </div>
                    )}

                    {activeTab === 'plan' && (
                        <div className="animate-fadeIn">
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                <span className="text-xs text-gray-500">Tratamiento, seguimiento y recomendaciones</span>
                            </label>
                            <textarea
                                className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700 min-h-[150px]"
                                placeholder="Medicamentos, estudios solicitados, orientaciones al paciente..."
                                value={soap.plan || ''}
                                onChange={(e) => handleChange('plan', e.target.value)}
                            />
                        </div>
                    )}
                </div>
            </div>

            {/* Physical Exam Section */}
            <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-sm border border-gray-100 dark:border-gray-700">
                <div className="flex justify-between items-center mb-4">
                    <h3 className="text-lg font-bold text-gray-900 dark:text-white flex items-center gap-2">
                        🩺 Examen Físico por Sistemas
                    </h3>
                    <button
                        type="button"
                        onClick={handleAddExam}
                        className="px-3 py-1.5 bg-blue-100 text-blue-700 rounded-lg text-sm font-medium hover:bg-blue-200 transition-colors"
                    >
                        + Agregar Sistema
                    </button>
                </div>

                {examenFisico.length === 0 ? (
                    <div className="text-center p-6 bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-dashed border-gray-300 dark:border-gray-600 text-gray-500">
                        No hay hallazgos registrados. Agregue un sistema para comenzar.
                    </div>
                ) : (
                    <div className="space-y-3">
                        {examenFisico.map((item, index) => (
                            <div key={index} className="flex gap-3 items-start p-3 bg-gray-50 dark:bg-gray-700/30 rounded-lg border border-gray-200 dark:border-gray-600">
                                <div className="w-1/3">
                                    <input
                                        list={`sistemas-${index}`}
                                        className="w-full p-2 border border-gray-300 rounded-md text-sm"
                                        placeholder="Sistema (ej. Cardiovascular)"
                                        value={item.sistema}
                                        onChange={(e) => handleExamChange(index, 'sistema', e.target.value)}
                                    />
                                    <datalist id={`sistemas-${index}`}>
                                        {sistemasComunes.map(s => <option key={s} value={s} />)}
                                    </datalist>
                                </div>
                                <div className="flex-1">
                                    <textarea
                                        className="w-full p-2 border border-gray-300 rounded-md text-sm min-h-[60px]"
                                        placeholder="Hallazgos (ej. Ruidos cardiacos rítmicos sin soplos...)"
                                        value={item.hallazgos}
                                        onChange={(e) => handleExamChange(index, 'hallazgos', e.target.value)}
                                    />
                                </div>
                                <button
                                    type="button"
                                    onClick={() => handleRemoveExam(index)}
                                    className="text-red-500 hover:text-red-700 p-1"
                                    title="Eliminar"
                                >
                                    🗑️
                                </button>
                            </div>
                        ))}
                    </div>
                )}
            </div>
        </div>
    );
}
