import React, { useState, useEffect } from 'react';
import { Link, useNavigate, useLocation } from 'react-router-dom';
import axios from 'axios';
import { Sun, Moon, Menu, LogOut, User, FlaskConical, LayoutDashboard, Stethoscope } from 'lucide-react';

interface Paciente {
    nombre: string;
    nombre_completo: string;
    email: string;
    numero_historia: string;
}

interface LayoutPacienteProps {
    children: React.ReactNode;
}

const LayoutPaciente: React.FC<LayoutPacienteProps> = ({ children }) => {
    const navigate = useNavigate();
    const location = useLocation();
    const [paciente, setPaciente] = useState<Paciente | null>(null);
    const [sidebarOpen, setSidebarOpen] = useState(false);

    // Inicializar estado directamente desde localStorage para evitar flash y pérdida de estado
    const [isDarkMode, setIsDarkMode] = useState(() => {
        const savedMode = localStorage.getItem('paciente_dark_mode');
        return savedMode === 'true';
    });

    useEffect(() => {
        // Cargar datos del paciente del localStorage
        const pacienteData = localStorage.getItem('paciente');
        if (pacienteData) {
            try {
                setPaciente(JSON.parse(pacienteData));
            } catch (e) {
                console.error("Error parsing paciente data", e);
            }
        }
    }, []);

    useEffect(() => {
        // Guardar preferencia en localStorage y aplicar al HTML cada vez que cambie
        localStorage.setItem('paciente_dark_mode', isDarkMode.toString());
        if (isDarkMode) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    }, [isDarkMode]);

    const toggleDarkMode = () => {
        setIsDarkMode(!isDarkMode);
    };

    const handleLogout = async () => {
        try {
            const token = localStorage.getItem('paciente_token');
            if (token) {
                await axios.post('/paciente/logout', {}, {
                    headers: { 'Authorization': `Bearer ${token}` }
                });
            }
        } catch (error) {
            console.error('Error al cerrar sesión:', error);
        } finally {
            localStorage.removeItem('paciente_token');
            localStorage.removeItem('paciente');
            localStorage.removeItem('paciente_id');
            navigate('/paciente/login');
        }
    };

    const menuItems = [
        { path: '/paciente/dashboard', icon: LayoutDashboard, label: 'Dashboard' },
        { path: '/paciente/historial', icon: Stethoscope, label: 'Mi Historial' },
        { path: '/paciente/examenes', icon: FlaskConical, label: 'Mis Exámenes' },
        { path: '/paciente/perfil', icon: User, label: 'Mi Perfil' },
    ];

    const isActive = (path: string) => location.pathname === path;

    return (
        // Quitamos la clase 'dark' del wrapper y dejamos que actúe en el HTML
        <div className={`min-h-screen transition-all duration-300 ${isDarkMode ? 'bg-gradient-to-br from-gray-900 via-slate-900 to-zinc-900' : 'bg-gradient-to-br from-gray-50 via-blue-50 to-cyan-50'}`}>
            {/* Header */}
            <header className={`sticky top-0 z-40 transition-all duration-300 shadow-lg backdrop-blur-lg ${isDarkMode ? 'bg-gray-900/95 border-b border-gray-700/50' : 'bg-white/95 border-b border-gray-200/50'}`}>
                <div className="px-6 py-4">
                    <div className="flex items-center justify-between">
                        {/* Logo y Toggle */}
                        <div className="flex items-center space-x-4">
                            <button
                                onClick={() => setSidebarOpen(!sidebarOpen)}
                                className={`lg:hidden p-2 rounded-xl transition-all ${isDarkMode ? 'hover:bg-gray-800 text-gray-200' : 'hover:bg-gray-100 text-gray-700'}`}
                            >
                                <Menu className="w-6 h-6" />
                            </button>
                            <div className="flex items-center space-x-3">
                                <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-blue-600 to-cyan-600 flex items-center justify-center text-white text-xl font-bold shadow-lg">
                                    🏥
                                </div>
                                <div>
                                    <h1 className={`text-xl font-bold transition-colors bg-gradient-to-r from-blue-600 to-cyan-500 bg-clip-text text-transparent`}>
                                        Portal Paciente
                                    </h1>
                                    <p className={`text-xs transition-colors ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>Cibertronia Salud</p>
                                </div>
                            </div>
                        </div>

                        {/* Usuario y Toggle Dark Mode */}
                        <div className="flex items-center space-x-4">
                            {/* Toggle Modo Oscuro */}
                            <button
                                onClick={toggleDarkMode}
                                className={`p-2 rounded-xl transition-all hover:scale-110 ${isDarkMode ? 'hover:bg-gray-800 text-yellow-400' : 'hover:bg-gray-100 text-gray-600'}`}
                                title={isDarkMode ? 'Cambiar a modo claro' : 'Cambiar a modo oscuro'}
                            >
                                {isDarkMode ? <Sun className="w-5 h-5" /> : <Moon className="w-5 h-5" />}
                            </button>

                            <div className={`hidden md:flex items-center gap-3 px-4 py-2 rounded-xl border ${isDarkMode ? 'bg-gray-800 border-gray-700' : 'bg-gray-100 border-gray-200'}`}>
                                <div className="text-right">
                                    <p className={`text-sm font-semibold transition-colors ${isDarkMode ? 'text-white' : 'text-gray-900'}`}>{paciente?.nombre_completo || 'Cargando...'}</p>
                                    <p className={`text-xs transition-colors ${isDarkMode ? 'text-gray-400' : 'text-gray-600'}`}>{paciente?.numero_historia}</p>
                                </div>
                                <div className="w-8 h-8 rounded-full bg-gradient-to-br from-blue-500 to-cyan-500 flex items-center justify-center text-white font-bold text-sm">
                                    {paciente?.nombre ? paciente.nombre.charAt(0) : 'P'}
                                </div>
                            </div>

                            <button
                                onClick={handleLogout}
                                className={`p-2 rounded-xl transition-all hover:scale-110 text-white shadow-lg bg-red-600 hover:bg-red-700`}
                                title="Cerrar Sesión"
                            >
                                <LogOut className="w-5 h-5" />
                            </button>
                        </div>
                    </div>
                </div>
            </header>

            <div className="flex">
                {/* Sidebar */}
                <aside className={`
                    ${sidebarOpen ? 'translate-x-0' : '-translate-x-full'}
                    lg:translate-x-0
                    fixed lg:sticky top-[88px] left-0 z-30
                    w-72 h-[calc(100vh-88px)]
                    backdrop-blur-lg
                    shadow-2xl
                    transition-all duration-300 ease-in-out
                    overflow-y-auto
                    ${isDarkMode ? 'bg-gray-900/95 border-r border-gray-700/50' : 'bg-white/95 border-r border-gray-200/50'}
                `}>
                    <nav className="p-4 space-y-2">
                        {menuItems.map((item) => {
                            const Icon = item.icon;
                            return (
                                <Link
                                    key={item.path}
                                    to={item.path}
                                    onClick={() => setSidebarOpen(false)}
                                    className={`
                                        flex items-center space-x-3 px-4 py-3 rounded-xl font-medium transition-all group
                                        ${isActive(item.path)
                                            ? 'bg-gradient-to-r from-blue-600 to-cyan-600 text-white shadow-lg transform scale-105'
                                            : isDarkMode
                                                ? 'text-gray-300 hover:bg-gray-800 hover:translate-x-1'
                                                : 'text-gray-700 hover:bg-gray-100 hover:translate-x-1'
                                        }
                                    `}
                                >
                                    <Icon className={`w-5 h-5 ${isActive(item.path) ? 'text-white' : (isDarkMode ? 'text-gray-400 group-hover:text-white' : 'text-gray-500 group-hover:text-blue-600')}`} />
                                    <span>{item.label}</span>
                                </Link>
                            );
                        })}
                    </nav>

                    {/* Info del Paciente en Sidebar */}
                    <div className={`mx-4 mt-auto mb-6 rounded-xl p-4 border ${isDarkMode ? 'bg-gray-800/50 border-gray-700' : 'bg-blue-50/50 border-blue-100'}`}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-100 rounded-lg text-blue-600">
                                <User className="w-4 h-4" />
                            </div>
                            <div>
                                <p className={`text-xs font-bold uppercase ${isDarkMode ? 'text-gray-400' : 'text-gray-500'}`}>Tu Cuenta</p>
                                <p className={`text-xs ${isDarkMode ? 'text-gray-500' : 'text-gray-400'}`}>Paciente Registrado</p>
                            </div>
                        </div>
                        <p className={`text-sm font-medium truncate ${isDarkMode ? 'text-gray-300' : 'text-gray-700'}`}>{paciente?.email}</p>
                    </div>
                </aside>

                {/* Overlay para móvil */}
                {sidebarOpen && (
                    <div
                        className="fixed inset-0 bg-black/60 backdrop-blur-sm z-20 lg:hidden"
                        onClick={() => setSidebarOpen(false)}
                    />
                )}

                {/* Main Content */}
                <main className="flex-1 p-6 lg:p-8">
                    <div className="max-w-7xl mx-auto animate-fade-in-up">
                        {children}
                    </div>
                </main>
            </div>
        </div>
    );
};

export default LayoutPaciente;
