import { useState, useEffect } from 'react';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

interface CalendarEvent {
    id: number;
    title: string;
    event_type: string;
    start_at: string;
    end_at?: string;
    project?: { id: number; name: string };
}

const eventTypeConfig: Record<string, { label: string; emoji: string; class: string }> = {
    sprint_planning: { label: 'Sprint Planning', emoji: '📋', class: 'badge-primary' },
    daily: { label: 'Daily', emoji: '☀️', class: 'badge-info' },
    review: { label: 'Review', emoji: '👀', class: 'badge-success' },
    retro: { label: 'Retrospectiva', emoji: '🔄', class: 'badge-warning' },
    release: { label: 'Release', emoji: '🚀', class: 'badge-danger' },
    deadline: { label: 'Deadline', emoji: '⏰', class: 'badge-danger' },
    meeting: { label: 'Reunión', emoji: '🤝', class: 'badge-default' },
};

export default function AppDevCalendar() {
    const [events, setEvents] = useState<CalendarEvent[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        fetchEvents();
    }, []);

    const fetchEvents = async () => {
        try {
            const from = new Date();
            from.setDate(from.getDate() - 30);
            const to = new Date();
            to.setDate(to.getDate() + 60);

            const res = await axios.get('/api/app-dev/calendar', {
                params: {
                    from: from.toISOString().split('T')[0],
                    to: to.toISOString().split('T')[0],
                }
            });
            setEvents(res.data);
        } catch (error) {
            console.error('Error cargando eventos:', error);
        } finally {
            setLoading(false);
        }
    };

    if (loading) return <div className="p-8 text-center">Cargando calendario...</div>;

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Calendario de Eventos"
                subtitle="Eventos y fechas importantes de los proyectos"
            />

            <div className="table-container">
                <div className="table-header">
                    <h2 className="table-title">📅 Próximos Eventos</h2>
                </div>
                <div className="table-wrapper">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Fecha</th>
                                <th>Evento</th>
                                <th>Tipo</th>
                                <th>Proyecto</th>
                            </tr>
                        </thead>
                        <tbody>
                            {events.length === 0 ? (
                                <tr>
                                    <td colSpan={4} style={{ textAlign: 'center', padding: '2rem', color: 'var(--text-muted)' }}>
                                        No hay eventos programados
                                    </td>
                                </tr>
                            ) : (
                                events.map((event) => (
                                    <tr key={event.id}>
                                        <td>{new Date(event.start_at).toLocaleDateString()}</td>
                                        <td style={{ fontWeight: 600 }}>{event.title}</td>
                                        <td>
                                            <span className={`badge ${eventTypeConfig[event.event_type]?.class}`}>
                                                {eventTypeConfig[event.event_type]?.emoji}{' '}
                                                {eventTypeConfig[event.event_type]?.label}
                                            </span>
                                        </td>
                                        <td>{event.project?.name || 'N/A'}</td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
}
