import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

interface Project {
    id: number;
    name: string;
    code: string;
    app_type: string;
    status: string;
    start_date: string;
    due_date?: string;
    client?: { id: number; name: string };
    plan?: { id: number; name: string };
}

const statusConfig: Record<string, { label: string; class: string; emoji: string }> = {
    discovery: { label: 'Descubrimiento', class: 'badge-default', emoji: '🔍' },
    in_progress: { label: 'En Progreso', class: 'badge-primary', emoji: '🚀' },
    paused: { label: 'Pausado', class: 'badge-warning', emoji: '⏸️' },
    qa: { label: 'QA', class: 'badge-info', emoji: '🧪' },
    delivered: { label: 'Entregado', class: 'badge-success', emoji: '✅' },
    closed: { label: 'Cerrado', class: 'badge-default', emoji: '🔒' },
};

const appTypeConfig: Record<string, { label: string; emoji: string }> = {
    web: { label: 'Web', emoji: '🌐' },
    desktop: { label: 'Desktop', emoji: '💻' },
    android: { label: 'Android', emoji: '🤖' },
    ios: { label: 'iOS', emoji: '🍎' },
    cross_platform: { label: 'Multiplataforma', emoji: '📱' },
};

export default function AppDevProjects() {
    const [projects, setProjects] = useState<Project[]>([]);
    const [clients, setClients] = useState<any[]>([]);
    const [plans, setPlans] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [formData, setFormData] = useState({
        client_id: '',
        plan_id: '',
        name: '',
        app_type: 'web',
        start_date: new Date().toISOString().split('T')[0],
        due_date: '',
        notes: '',
    });

    useEffect(() => {
        fetchData();
    }, []);

    useEffect(() => {
        const handleEscape = (e: KeyboardEvent) => {
            if (e.key === 'Escape') setShowModal(false);
        };
        if (showModal) {
            document.addEventListener('keydown', handleEscape);
            return () => document.removeEventListener('keydown', handleEscape);
        }
    }, [showModal]);


    const fetchData = async () => {
        try {
            const [projectsRes, clientsRes, plansRes] = await Promise.all([
                axios.get('/api/app-dev/projects'),
                axios.get('/api/clients'),
                axios.get('/api/app-dev/plans'),
            ]);
            setProjects(Array.isArray(projectsRes.data) ? projectsRes.data : (projectsRes.data.data || []));
            setClients(Array.isArray(clientsRes.data) ? clientsRes.data : (clientsRes.data.data || []));
            setPlans(Array.isArray(plansRes.data) ? plansRes.data : (plansRes.data.data || []));

        } catch (error) {
            console.error('Error cargando datos:', error);
            setProjects([]);
            setClients([]);
            setPlans([]);
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            await axios.post('/api/app-dev/projects', formData);
            setShowModal(false);
            fetchData();
            setFormData({
                client_id: '',
                plan_id: '',
                name: '',
                app_type: 'web',
                start_date: new Date().toISOString().split('T')[0],
                due_date: '',
                notes: '',
            });
        } catch (error) {
            console.error('Error creando proyecto:', error);
            alert('Error al crear el proyecto');
        }
    };

    const handleOpenModal = () => {
        console.log('Abriendo modal...', { clients: clients.length, plans: plans.length });
        if (clients.length === 0) {
            alert('Primero debes crear al menos un cliente');
            return;
        }
        setShowModal(true);
        console.log('Modal abierto, showModal:', true);
    };


    if (loading) return <div className="p-8 text-center">Cargando proyectos...</div>;


    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Proyectos de Desarrollo"
                subtitle="Gestión de proyectos de aplicaciones"
                actions={
                    <button onClick={handleOpenModal} className="btn btn-primary">
                        ➕ Nuevo Proyecto
                    </button>
                }
            />


            <div className="table-container">
                <div className="table-header">
                    <h2 className="table-title">📱 Proyectos Activos</h2>
                </div>
                <div className="table-wrapper">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Código</th>
                                <th>Proyecto</th>
                                <th>Cliente</th>
                                <th>Tipo</th>
                                <th>Estado</th>
                                <th>Inicio</th>
                            </tr>
                        </thead>
                        <tbody>
                            {projects.length === 0 ? (
                                <tr>
                                    <td colSpan={6} style={{ textAlign: 'center', padding: '2rem', color: 'var(--text-muted)' }}>
                                        No hay proyectos registrados
                                    </td>
                                </tr>
                            ) : (
                                projects.map((project) => (
                                    <tr key={project.id}>
                                        <td>
                                            <span className="badge badge-default">{project.code}</span>
                                        </td>
                                        <td style={{ fontWeight: 600 }}>{project.name}</td>
                                        <td>{project.client?.name || 'N/A'}</td>
                                        <td>
                                            <span>
                                                {appTypeConfig[project.app_type]?.emoji}{' '}
                                                {appTypeConfig[project.app_type]?.label}
                                            </span>
                                        </td>
                                        <td>
                                            <span className={`badge ${statusConfig[project.status]?.class}`}>
                                                {statusConfig[project.status]?.emoji}{' '}
                                                {statusConfig[project.status]?.label}
                                            </span>
                                        </td>
                                        <td>{project.start_date}</td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Modal Crear Proyecto */}
            {/* Modal Crear Proyecto */}
            {showModal && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setShowModal(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)', position: 'relative' }}>
                            <button
                                type="button"
                                onClick={() => setShowModal(false)}
                                style={{
                                    position: 'absolute',
                                    top: '1rem',
                                    right: '1rem',
                                    background: 'transparent',
                                    border: 'none',
                                    fontSize: '1.5rem',
                                    cursor: 'pointer',
                                    color: 'var(--text-muted)',
                                    padding: '0.25rem',
                                    lineHeight: 1,
                                }}
                            >
                                ✕
                            </button>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                📱 Nuevo Proyecto
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Crea un nuevo proyecto de desarrollo de aplicación
                            </p>
                        </div>

                        <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Cliente
                                    </label>
                                    <select
                                        required
                                        className="input"
                                        value={formData.client_id}
                                        onChange={(e) => setFormData({ ...formData, client_id: e.target.value })}
                                    >
                                        <option value="">Seleccionar cliente...</option>
                                        {clients.map(c => (
                                            <option key={c.id} value={c.id}>{c.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Nombre del proyecto
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        className="input"
                                        placeholder="Ej: App de Delivery XYZ"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Tipo de aplicación
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.app_type}
                                        onChange={(e) => setFormData({ ...formData, app_type: e.target.value })}
                                    >
                                        <option value="web">🌐 Web</option>
                                        <option value="desktop">💻 Desktop</option>
                                        <option value="android">🤖 Android</option>
                                        <option value="ios">🍎 iOS</option>
                                        <option value="cross_platform">📱 Multiplataforma</option>
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Plan (opcional)
                                    </label>
                                    <select
                                        className="input"
                                        value={formData.plan_id}
                                        onChange={(e) => setFormData({ ...formData, plan_id: e.target.value })}
                                    >
                                        <option value="">Sin plan predefinido</option>
                                        {plans.map(p => (
                                            <option key={p.id} value={p.id}>{p.name}</option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Fecha de inicio
                                    </label>
                                    <input
                                        type="date"
                                        required
                                        className="input"
                                        value={formData.start_date}
                                        onChange={(e) => setFormData({ ...formData, start_date: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Notas
                                    </label>
                                    <textarea
                                        className="input"
                                        rows={3}
                                        placeholder="Información adicional del proyecto..."
                                        value={formData.notes}
                                        onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
                                    />
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setShowModal(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        🚀 Crear Proyecto
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}

        </div>

    );
}
