import { useEffect, useState, CSSProperties } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import NewServiceModal from '../components/NewServiceModal';

interface Metrics {
    monthly_revenue: number;
    domains_expiring_soon: number;
    active_projects: number;
    total_clients: number;
    services_expiring_this_month: number;
    pending_invoices: number;
    overdue_invoices: number;
    open_tickets: number;
    revenue_change: number;
    clients_change: number;
}

interface ServicesByCategory {
    category: string;
    name: string;
    icon: string;
    total: number;
    active: number;
}

interface UpcomingRenewal {
    id: number;
    name: string;
    client_name: string;
    type: string;
    expiry_date: string;
    status: 'active' | 'expiring_soon' | 'expired';
    days_until_expiry: number;
    price: number;
}

interface Invoice {
    id: number;
    number: string;
    client_name: string;
    amount: number;
    due_date: string;
    status: 'pending' | 'overdue' | 'paid';
    days_overdue: number;
}

interface Ticket {
    id: number;
    subject: string;
    client_name: string;
    priority: 'low' | 'medium' | 'high' | 'critical';
    status: 'open' | 'in_progress' | 'waiting' | 'resolved';
    created_at: string;
}

interface Activity {
    id: number;
    type: 'payment' | 'renewal' | 'ticket' | 'new_client' | 'new_service';
    description: string;
    client_name: string;
    timestamp: string;
    amount?: number;
}

// Mock data for demo - Ready for MariaDB connection
const mockMetrics: Metrics = {
    monthly_revenue: 15840.00,
    domains_expiring_soon: 5,
    active_projects: 12,
    total_clients: 47,
    services_expiring_this_month: 8,
    pending_invoices: 4,
    overdue_invoices: 2,
    open_tickets: 7,
    revenue_change: 12.5,
    clients_change: 8,
};

const mockServicesByCategory: ServicesByCategory[] = [
    { category: 'domains', name: 'Dominios', icon: '🌐', total: 24, active: 21 },
    { category: 'hosting', name: 'Hosting', icon: '🖥️', total: 15, active: 13 },
    { category: 'marketing', name: 'Agencia de Marketing', icon: '📣', total: 8, active: 8 },
    { category: 'web_design', name: 'Diseño Web', icon: '🎨', total: 6, active: 4 },
    { category: 'software_dev', name: 'Desarrollo de Software', icon: '💻', total: 5, active: 5 },
    { category: 'app_dev', name: 'Desarrollo de Apps', icon: '📱', total: 4, active: 3 },
    { category: 'maintenance', name: 'Mantenimiento', icon: '🔧', total: 3, active: 3 },
    { category: 'other', name: 'Otros Servicios', icon: '📦', total: 7, active: 5 },
];

const mockUpcomingRenewals: UpcomingRenewal[] = [
    { id: 1, name: 'acme-corp.com', client_name: 'Acme Corporation', type: 'domain', expiry_date: '2026-01-28', status: 'expiring_soon', days_until_expiry: 3, price: 12.99 },
    { id: 2, name: 'Hosting Empresarial Pro', client_name: 'Beta Technologies', type: 'hosting', expiry_date: '2026-02-15', status: 'active', days_until_expiry: 21, price: 240.00 },
    { id: 3, name: 'gammastore.com', client_name: 'Gamma Retail SA', type: 'domain', expiry_date: '2026-01-20', status: 'expired', days_until_expiry: -5, price: 14.99 },
    { id: 4, name: 'Licencia Microsoft 365', client_name: 'Delta Industries', type: 'license', expiry_date: '2026-02-01', status: 'expiring_soon', days_until_expiry: 7, price: 150.00 },
    { id: 5, name: 'Gestión Redes Sociales', client_name: 'Sigma Corp', type: 'support', expiry_date: '2026-01-31', status: 'expiring_soon', days_until_expiry: 6, price: 450.00 },
];

const mockPendingInvoices: Invoice[] = [
    { id: 1, number: 'INV-2026-00045', client_name: 'Gamma Retail SA', amount: 320.00, due_date: '2025-12-31', status: 'overdue', days_overdue: 25 },
    { id: 2, number: 'INV-2026-00051', client_name: 'Delta Industries', amount: 1250.00, due_date: '2026-01-15', status: 'overdue', days_overdue: 10 },
    { id: 3, number: 'INV-2026-00058', client_name: 'Acme Corporation', amount: 450.00, due_date: '2026-02-01', status: 'pending', days_overdue: 0 },
    { id: 4, number: 'INV-2026-00060', client_name: 'Beta Technologies', amount: 890.00, due_date: '2026-02-10', status: 'pending', days_overdue: 0 },
];

const mockOpenTickets: Ticket[] = [
    { id: 1, subject: 'Página web no carga correctamente', client_name: 'Acme Corporation', priority: 'critical', status: 'open', created_at: '2026-01-25T10:30:00' },
    { id: 2, subject: 'Error en formulario de contacto', client_name: 'Beta Technologies', priority: 'high', status: 'in_progress', created_at: '2026-01-24T14:20:00' },
    { id: 3, subject: 'Solicitud de backup de base de datos', client_name: 'Gamma Retail SA', priority: 'medium', status: 'waiting', created_at: '2026-01-23T09:15:00' },
    { id: 4, subject: 'Actualizar certificado SSL', client_name: 'Delta Industries', priority: 'high', status: 'open', created_at: '2026-01-25T08:00:00' },
    { id: 5, subject: 'Configurar correos corporativos', client_name: 'Epsilon Tech', priority: 'low', status: 'in_progress', created_at: '2026-01-22T16:45:00' },
];

const mockRecentActivity: Activity[] = [
    { id: 1, type: 'payment', description: 'Pago recibido - Hosting anual', client_name: 'Beta Technologies', timestamp: '2026-01-25T18:30:00', amount: 240.00 },
    { id: 2, type: 'renewal', description: 'Renovación automática de dominio', client_name: 'Lambda Digital', timestamp: '2026-01-25T15:00:00' },
    { id: 3, type: 'ticket', description: 'Nuevo ticket de soporte', client_name: 'Acme Corporation', timestamp: '2026-01-25T10:30:00' },
    { id: 4, type: 'new_service', description: 'Nuevo servicio contratado - SEO', client_name: 'Upsilon Co', timestamp: '2026-01-24T14:00:00', amount: 3600.00 },
    { id: 5, type: 'new_client', description: 'Nuevo cliente registrado', client_name: 'Omega Ventures', timestamp: '2026-01-24T11:20:00' },
    { id: 6, type: 'payment', description: 'Pago recibido - Desarrollo web', client_name: 'Theta Design', timestamp: '2026-01-23T16:45:00', amount: 2800.00 },
];

const typeIcons: Record<string, string> = {
    domain: '🌐',
    hosting: '🖥️',
    license: '📄',
    development: '💻',
    support: '🛠️',
};

const priorityConfig: Record<string, { label: string; color: string }> = {
    low: { label: 'Baja', color: '#64748b' },
    medium: { label: 'Media', color: '#f59e0b' },
    high: { label: 'Alta', color: '#ef4444' },
    critical: { label: 'Crítica', color: '#dc2626' },
};

const ticketStatusConfig: Record<string, { label: string; class: string }> = {
    open: { label: 'Abierto', class: 'badge-danger' },
    in_progress: { label: 'En Progreso', class: 'badge-primary' },
    waiting: { label: 'Esperando', class: 'badge-warning' },
    resolved: { label: 'Resuelto', class: 'badge-success' },
};

const activityIcons: Record<string, string> = {
    payment: '💵',
    renewal: '🔄',
    ticket: '🎫',
    new_client: '👤',
    new_service: '🚀',
};

export default function Dashboard() {
    const [metrics, setMetrics] = useState<Metrics>(mockMetrics);
    const [servicesByCategory] = useState<ServicesByCategory[]>(mockServicesByCategory);
    const [upcomingRenewals, setUpcomingRenewals] = useState<UpcomingRenewal[]>(mockUpcomingRenewals);
    const [pendingInvoices] = useState<Invoice[]>(mockPendingInvoices);
    const [openTickets] = useState<Ticket[]>(mockOpenTickets);
    const [recentActivity] = useState<Activity[]>(mockRecentActivity);
    const [loading, setLoading] = useState(false);
    const [showNewServiceModal, setShowNewServiceModal] = useState(false);

    useEffect(() => {
        setLoading(true);
        const fetchData = async () => {
            try {
                const [metricsRes, activityRes] = await Promise.all([
                    axios.get('/api/dashboard/metrics'),
                    axios.get('/api/dashboard/activity')
                ]);

                setMetrics({
                    ...mockMetrics,
                    ...metricsRes.data.data
                });

                const activityData = activityRes.data.data;
                const renewals = activityData.upcoming_renewals?.map((s: any) => ({
                    id: s.id,
                    name: s.name,
                    client_name: s.client?.name || 'N/A',
                    type: s.type,
                    expiry_date: s.expiry_date,
                    status: s.status,
                    days_until_expiry: Math.ceil((new Date(s.expiry_date).getTime() - new Date().getTime()) / (1000 * 3600 * 24)),
                    price: s.price || 0,
                })) || mockUpcomingRenewals;

                setUpcomingRenewals(renewals);
            } catch (error) {
                console.error("Error fetching dashboard data:", error);
            } finally {
                setLoading(false);
            }
        };

        fetchData();
    }, []);

    // Calculate totals
    const totalActiveServices = servicesByCategory.reduce((sum, cat) => sum + cat.active, 0);
    const totalServices = servicesByCategory.reduce((sum, cat) => sum + cat.total, 0);

    const metricCards = [
        {
            icon: '💵',
            label: 'Ingresos del mes',
            value: `$${metrics.monthly_revenue.toLocaleString('es-ES', { minimumFractionDigits: 2 })}`,
            change: metrics.revenue_change,
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
        {
            icon: '🚀',
            label: 'Servicios activos',
            value: totalActiveServices.toString(),
            subtext: `de ${totalServices} totales`,
            gradient: 'linear-gradient(135deg, #3b82f6, #2563eb)',
            glow: 'rgba(59, 130, 246, 0.4)',
        },
        {
            icon: '⏰',
            label: 'Renovaciones próximas',
            value: metrics.services_expiring_this_month.toString(),
            gradient: 'linear-gradient(135deg, #f59e0b, #d97706)',
            glow: 'rgba(245, 158, 11, 0.4)',
            alert: metrics.services_expiring_this_month > 0,
        },
        {
            icon: '👥',
            label: 'Total de clientes',
            value: metrics.total_clients.toString(),
            change: metrics.clients_change,
            gradient: 'linear-gradient(135deg, #8b5cf6, #7c3aed)',
            glow: 'rgba(139, 92, 246, 0.4)',
        },
        {
            icon: '📄',
            label: 'Facturas pendientes',
            value: metrics.pending_invoices.toString(),
            subtext: `${metrics.overdue_invoices} vencidas`,
            gradient: 'linear-gradient(135deg, #06b6d4, #0891b2)',
            glow: 'rgba(6, 182, 212, 0.4)',
            alert: metrics.overdue_invoices > 0,
        },
        {
            icon: '🎫',
            label: 'Tickets abiertos',
            value: metrics.open_tickets.toString(),
            gradient: 'linear-gradient(135deg, #ec4899, #db2777)',
            glow: 'rgba(236, 72, 153, 0.4)',
            alert: metrics.open_tickets > 5,
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando métricas...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            {/* Page Header */}
            <div className="page-header">
                <div>
                    <h1 className="page-title">Dashboard Principal</h1>
                    <p className="page-subtitle">Resumen ejecutivo de tu agencia</p>
                </div>
                <div style={{ display: 'flex', gap: '0.75rem' }}>
                    <Link to="/app/services" className="btn btn-secondary">
                        📋 Ver Servicios
                    </Link>
                    <button className="btn btn-primary" onClick={() => setShowNewServiceModal(true)}>
                        ➕ Nuevo Servicio
                    </button>
                </div>
            </div>

            {/* Quick Stats Tags */}
            <div className="stats-row" style={{ display: 'flex', gap: '0.75rem', marginBottom: '2rem', flexWrap: 'wrap' }}>
                <div className="stat-tag">
                    <span>🗓️</span>
                    <span>Hoy:</span>
                    <span className="stat-tag-value">{new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' })}</span>
                </div>
                <div className="stat-tag">
                    <span>⚡</span>
                    <span>Alertas activas:</span>
                    <span className="stat-tag-value" style={{ color: 'var(--color-warning)' }}>{metrics.domains_expiring_soon + metrics.overdue_invoices}</span>
                </div>
                <div className="stat-tag">
                    <span>✅</span>
                    <span>Servicios totales:</span>
                    <span className="stat-tag-value">{totalServices}</span>
                </div>
            </div>

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                            {card.change !== undefined && (
                                <div className={`metric-change ${card.change >= 0 ? 'positive' : 'negative'}`}>
                                    {card.change >= 0 ? '↑' : '↓'} {Math.abs(card.change)}% vs. mes anterior
                                </div>
                            )}
                            {card.alert && (
                                <div className="metric-change negative" style={{ background: 'rgba(239, 68, 68, 0.1)', color: 'var(--color-danger)' }}>
                                    ⚠️ Requiere atención
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Services by Category */}
            <div style={{ marginTop: '2.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">📊 Servicios Activos por Categoría</h2>
                        <Link to="/app/services" className="btn btn-ghost btn-sm">
                            Ver todos →
                        </Link>
                    </div>
                    <div style={{ padding: '1rem', display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(180px, 1fr))', gap: '0.75rem' }}>
                        {servicesByCategory.map((cat) => (
                            <Link
                                key={cat.category}
                                to={`/app/services?category=${cat.category}`}
                                style={{
                                    display: 'flex',
                                    alignItems: 'center',
                                    gap: '0.75rem',
                                    padding: '1rem',
                                    background: 'var(--glass-bg)',
                                    borderRadius: '12px',
                                    textDecoration: 'none',
                                    color: 'inherit',
                                    transition: 'all 0.2s ease',
                                    border: '1px solid var(--border-color)',
                                }}
                            >
                                <div style={{
                                    width: '40px',
                                    height: '40px',
                                    borderRadius: '10px',
                                    background: 'var(--glass-bg-hover)',
                                    display: 'flex',
                                    alignItems: 'center',
                                    justifyContent: 'center',
                                    fontSize: '1.25rem',
                                }}>
                                    {cat.icon}
                                </div>
                                <div style={{ flex: 1 }}>
                                    <div style={{ fontSize: '0.8rem', color: 'var(--text-muted)', marginBottom: '0.125rem' }}>{cat.name}</div>
                                    <div style={{ display: 'flex', alignItems: 'baseline', gap: '0.25rem' }}>
                                        <span style={{ fontSize: '1.25rem', fontWeight: 700, color: 'var(--text-primary)' }}>{cat.active}</span>
                                        <span style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>/ {cat.total}</span>
                                    </div>
                                </div>
                            </Link>
                        ))}
                    </div>
                </div>
            </div>

            {/* Two Column Layout */}
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(400px, 1fr))', gap: '1.5rem', marginTop: '1.5rem' }}>
                {/* Upcoming Renewals */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🔔 Próximas Renovaciones</h2>
                        <Link to="/app/calendar" className="btn btn-ghost btn-sm">
                            Ver calendario →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Servicio</th>
                                    <th>Cliente</th>
                                    <th>Vence</th>
                                    <th>Días</th>
                                    <th>Acción</th>
                                </tr>
                            </thead>
                            <tbody>
                                {upcomingRenewals.slice(0, 5).map((service) => (
                                    <tr key={service.id}>
                                        <td>
                                            <div style={{ display: 'flex', alignItems: 'center', gap: '0.5rem' }}>
                                                <span>{typeIcons[service.type] || '📦'}</span>
                                                <div className="cell-main">{service.name}</div>
                                            </div>
                                        </td>
                                        <td style={{ fontSize: '0.85rem' }}>{service.client_name}</td>
                                        <td>
                                            {new Date(service.expiry_date).toLocaleDateString('es-ES', {
                                                day: '2-digit',
                                                month: 'short'
                                            })}
                                        </td>
                                        <td>
                                            <span style={{
                                                fontWeight: 700,
                                                padding: '0.2rem 0.5rem',
                                                borderRadius: '6px',
                                                fontSize: '0.85rem',
                                                background: service.days_until_expiry <= 0
                                                    ? 'rgba(239, 68, 68, 0.1)'
                                                    : service.days_until_expiry <= 7
                                                        ? 'rgba(245, 158, 11, 0.1)'
                                                        : 'rgba(16, 185, 129, 0.1)',
                                                color: service.days_until_expiry <= 0
                                                    ? 'var(--color-danger)'
                                                    : service.days_until_expiry <= 7
                                                        ? 'var(--color-warning)'
                                                        : 'var(--color-success)'
                                            }}>
                                                {service.days_until_expiry <= 0 ? service.days_until_expiry : `+${service.days_until_expiry}`}
                                            </span>
                                        </td>
                                        <td>
                                            <button className="btn btn-primary btn-sm" style={{ padding: '0.3rem 0.6rem' }}>
                                                Renovar
                                            </button>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>

                {/* Pending & Overdue Invoices */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">💳 Facturas Pendientes</h2>
                        <Link to="/app/billing" className="btn btn-ghost btn-sm">
                            Ver todas →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Factura</th>
                                    <th>Cliente</th>
                                    <th>Monto</th>
                                    <th>Estado</th>
                                    <th>Acción</th>
                                </tr>
                            </thead>
                            <tbody>
                                {pendingInvoices.slice(0, 4).map((invoice) => (
                                    <tr key={invoice.id}>
                                        <td>
                                            <div className="cell-main">{invoice.number}</div>
                                            <div className="cell-secondary">
                                                Vence: {new Date(invoice.due_date).toLocaleDateString('es-ES', { day: '2-digit', month: 'short' })}
                                            </div>
                                        </td>
                                        <td style={{ fontSize: '0.85rem' }}>{invoice.client_name}</td>
                                        <td style={{ fontWeight: 600 }}>
                                            ${invoice.amount.toLocaleString('es-ES', { minimumFractionDigits: 2 })}
                                        </td>
                                        <td>
                                            <span className={`badge ${invoice.status === 'overdue' ? 'badge-danger' : 'badge-warning'}`}>
                                                <span className="badge-dot"></span>
                                                {invoice.status === 'overdue' ? `Vencida (${invoice.days_overdue}d)` : 'Pendiente'}
                                            </span>
                                        </td>
                                        <td>
                                            <button className="btn-icon" title="Ver factura" style={{ width: '32px', height: '32px' }}>
                                                👁️
                                            </button>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            {/* Second Two Column Layout */}
            <div style={{ display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(400px, 1fr))', gap: '1.5rem', marginTop: '1.5rem' }}>
                {/* Open Tickets */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🎫 Incidencias Abiertas</h2>
                        <Link to="/app/tickets" className="btn btn-ghost btn-sm">
                            Ver todas →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Asunto</th>
                                    <th>Cliente</th>
                                    <th>Prioridad</th>
                                    <th>Estado</th>
                                </tr>
                            </thead>
                            <tbody>
                                {openTickets.slice(0, 5).map((ticket) => (
                                    <tr key={ticket.id}>
                                        <td>
                                            <div className="cell-main" style={{ maxWidth: '200px', overflow: 'hidden', textOverflow: 'ellipsis', whiteSpace: 'nowrap' }}>
                                                {ticket.subject}
                                            </div>
                                        </td>
                                        <td style={{ fontSize: '0.85rem' }}>{ticket.client_name}</td>
                                        <td>
                                            <span style={{
                                                display: 'inline-flex',
                                                alignItems: 'center',
                                                gap: '0.25rem',
                                                padding: '0.2rem 0.6rem',
                                                borderRadius: '6px',
                                                fontSize: '0.75rem',
                                                fontWeight: 600,
                                                background: `${priorityConfig[ticket.priority].color}15`,
                                                color: priorityConfig[ticket.priority].color,
                                            }}>
                                                {ticket.priority === 'critical' && '🔴'}
                                                {ticket.priority === 'high' && '🟠'}
                                                {ticket.priority === 'medium' && '🟡'}
                                                {ticket.priority === 'low' && '🟢'}
                                                {priorityConfig[ticket.priority].label}
                                            </span>
                                        </td>
                                        <td>
                                            <span className={`badge ${ticketStatusConfig[ticket.status].class}`}>
                                                <span className="badge-dot"></span>
                                                {ticketStatusConfig[ticket.status].label}
                                            </span>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>

                {/* Recent Activity */}
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">📝 Actividad Reciente</h2>
                    </div>
                    <div style={{ padding: '0.5rem' }}>
                        {recentActivity.map((activity) => (
                            <div
                                key={activity.id}
                                style={{
                                    display: 'flex',
                                    alignItems: 'flex-start',
                                    gap: '1rem',
                                    padding: '1rem',
                                    borderBottom: '1px solid var(--border-color)',
                                }}
                            >
                                <div style={{
                                    width: '40px',
                                    height: '40px',
                                    borderRadius: '10px',
                                    background: 'var(--glass-bg)',
                                    display: 'flex',
                                    alignItems: 'center',
                                    justifyContent: 'center',
                                    fontSize: '1.25rem',
                                }}>
                                    {activityIcons[activity.type]}
                                </div>
                                <div style={{ flex: 1 }}>
                                    <div style={{ fontWeight: 500, color: 'var(--text-primary)', marginBottom: '0.25rem' }}>
                                        {activity.description}
                                    </div>
                                    <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>
                                        {activity.client_name} • {new Date(activity.timestamp).toLocaleDateString('es-ES', {
                                            day: '2-digit',
                                            month: 'short',
                                            hour: '2-digit',
                                            minute: '2-digit'
                                        })}
                                    </div>
                                </div>
                                {activity.amount && (
                                    <div style={{ fontWeight: 700, color: 'var(--color-success)' }}>
                                        +${activity.amount.toLocaleString('es-ES')}
                                    </div>
                                )}
                            </div>
                        ))}
                    </div>
                </div>
            </div>

            {/* Quick Actions */}
            <div style={{ marginTop: '2rem', display: 'grid', gridTemplateColumns: 'repeat(auto-fit, minmax(220px, 1fr))', gap: '1rem' }}>
                <Link to="/app/calendar" className="card" style={{ textDecoration: 'none', display: 'flex', alignItems: 'center', gap: '1rem' }}>
                    <div style={{
                        width: '48px',
                        height: '48px',
                        borderRadius: '12px',
                        background: 'linear-gradient(135deg, #f59e0b, #d97706)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        fontSize: '1.5rem',
                    }}>
                        📅
                    </div>
                    <div>
                        <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Ver Agenda</div>
                        <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>Calendario de vencimientos</div>
                    </div>
                </Link>

                <Link to="/app/clients" className="card" style={{ textDecoration: 'none', display: 'flex', alignItems: 'center', gap: '1rem' }}>
                    <div style={{
                        width: '48px',
                        height: '48px',
                        borderRadius: '12px',
                        background: 'linear-gradient(135deg, #8b5cf6, #7c3aed)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        fontSize: '1.5rem',
                    }}>
                        👥
                    </div>
                    <div>
                        <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Gestionar Clientes</div>
                        <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>{metrics.total_clients} clientes</div>
                    </div>
                </Link>

                <Link to="/app/billing" className="card" style={{ textDecoration: 'none', display: 'flex', alignItems: 'center', gap: '1rem' }}>
                    <div style={{
                        width: '48px',
                        height: '48px',
                        borderRadius: '12px',
                        background: 'linear-gradient(135deg, #06b6d4, #0891b2)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        fontSize: '1.5rem',
                    }}>
                        💳
                    </div>
                    <div>
                        <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Facturación</div>
                        <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>{metrics.pending_invoices} pendientes</div>
                    </div>
                </Link>

                <Link to="/app/tickets" className="card" style={{ textDecoration: 'none', display: 'flex', alignItems: 'center', gap: '1rem' }}>
                    <div style={{
                        width: '48px',
                        height: '48px',
                        borderRadius: '12px',
                        background: 'linear-gradient(135deg, #ec4899, #db2777)',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center',
                        fontSize: '1.5rem',
                    }}>
                        🎫
                    </div>
                    <div>
                        <div style={{ fontWeight: 600, color: 'var(--text-primary)' }}>Soporte</div>
                        <div style={{ fontSize: '0.85rem', color: 'var(--text-muted)' }}>{metrics.open_tickets} tickets abiertos</div>
                    </div>
                </Link>
            </div>

            <NewServiceModal isOpen={showNewServiceModal} onClose={() => setShowNewServiceModal(false)} />
        </div>
    );
}
