import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';


export default function DocumentShow() {
    const { id } = useParams();
    const [document, setDocument] = useState<any>(null);
    const [loading, setLoading] = useState(true);

    // Upload State
    const [uploading, setUploading] = useState(false);

    // Send Email State
    const [showSendForm, setShowSendForm] = useState(false);
    const [sendData, setSendData] = useState({
        to: '',
        subject: '',
        message: 'Please find attached the document.',
        attach_latest: true
    });
    const [sending, setSending] = useState(false);

    useEffect(() => {
        loadDocument();
    }, [id]);

    const loadDocument = () => {
        axios.get(`/api/documents/${id}`)
            .then(res => {
                setDocument(res.data);
                setLoading(false);
                // Pre-fill subject
                setSendData(prev => ({ ...prev, subject: `Document: ${res.data.title}` }));
            })
            .catch(err => setLoading(false));
    };

    const handleUpload = async (e: any) => {
        const file = e.target.files[0];
        if (!file) return;

        const formData = new FormData();
        formData.append('file', file);

        setUploading(true);
        try {
            await axios.post(`/api/documents/${id}/upload`, formData, {
                headers: { 'Content-Type': 'multipart/form-data' }
            });
            loadDocument(); // refresh
            alert('Archivo subido con éxito');
        } catch (error) {
            alert('Falló la subida');
        } finally {
            setUploading(false);
        }
    };

    const handleSend = async (e: any) => {
        e.preventDefault();
        setSending(true);
        try {
            // Split emails by comma
            const toList = sendData.to.split(',').map(e => e.trim()).filter(e => e);

            await axios.post(`/api/documents/${id}/send`, {
                recipients: { to: toList },
                subject: sendData.subject,
                message: sendData.message,
                attach_latest: sendData.attach_latest
            });
            alert('¡Correo puesto en cola de envío!');
            setShowSendForm(false);
            loadDocument(); // refresh history
        } catch (error: any) {
            alert('Error al enviar: ' + (error.response?.data?.message || error.message));
        } finally {
            setSending(false);
        }
    };

    const handleDownload = (version: number) => {
        window.location.href = `/api/documents/${id}/download?version=${version}`;
    };

    if (loading) return <div className="p-10 text-center">Cargando...</div>;
    if (!document) return <div className="p-10 text-center">Documento no encontrado</div>;

    return (
        <div className="p-6">
            <div className="flex items-center justify-between mb-6">
                <div>
                    <Link to="/app/documents" className="text-gray-400 hover:text-white text-sm"> &larr; Volver a la Lista</Link>
                    <h1 className="text-2xl font-bold text-white mt-2">{document.title}</h1>
                    <div className="flex gap-2 mt-2">
                        <span className="badge bg-gray-700 text-gray-300 px-2 py-1 rounded text-xs">{document.type}</span>
                        <span className="badge bg-blue-900 text-blue-200 px-2 py-1 rounded text-xs">{document.status}</span>
                    </div>
                </div>
                <div className="flex gap-2">
                    <button
                        onClick={() => setShowSendForm(!showSendForm)}
                        className="btn bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded"
                    >
                        {showSendForm ? 'Cancelar Envío' : 'Enviar por Correo'}
                    </button>
                </div>
            </div>

            {/* SEND FORM */}
            {showSendForm && (
                <div className="card bg-gray-800 p-6 rounded-lg mb-6 border border-gray-700 animate-fade-in">
                    <h3 className="text-lg font-bold text-white mb-4">Enviar Documento</h3>
                    <form onSubmit={handleSend}>
                        <div className="mb-4">
                            <label className="block text-gray-400 text-sm mb-1">Destinatarios (separados por coma)</label>
                            <input
                                type="text"
                                className="w-full bg-gray-900 border border-gray-700 rounded p-2 text-white"
                                placeholder="cliente@ejemplo.com, jefe@ejemplo.com"
                                value={sendData.to}
                                onChange={e => setSendData({ ...sendData, to: e.target.value })}
                                required
                            />
                        </div>
                        <div className="mb-4">
                            <label className="block text-gray-400 text-sm mb-1">Asunto</label>
                            <input
                                type="text"
                                className="w-full bg-gray-900 border border-gray-700 rounded p-2 text-white"
                                value={sendData.subject}
                                onChange={e => setSendData({ ...sendData, subject: e.target.value })}
                                required
                            />
                        </div>
                        <div className="mb-4">
                            <label className="block text-gray-400 text-sm mb-1">Mensaje</label>
                            <textarea
                                className="w-full bg-gray-900 border border-gray-700 rounded p-2 text-white h-24"
                                value={sendData.message}
                                onChange={e => setSendData({ ...sendData, message: e.target.value })}
                                required
                            />
                        </div>
                        <div className="mb-4 flex items-center">
                            <input
                                type="checkbox"
                                id="attach"
                                checked={sendData.attach_latest}
                                onChange={e => setSendData({ ...sendData, attach_latest: e.target.checked })}
                                className="mr-2"
                            />
                            <label htmlFor="attach" className="text-gray-300 text-sm">Adjuntar último archivo (PDF/Doc)</label>
                        </div>
                        <div className="flex justify-end">
                            <button type="submit" disabled={sending} className="btn bg-green-600 hover:bg-green-500 text-white px-6 py-2 rounded">
                                {sending ? 'Enviando...' : 'Enviar Ahora'}
                            </button>
                        </div>
                    </form>
                </div>
            )}

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">

                {/* FILES */}
                <div className="lg:col-span-2 space-y-6">
                    <div className="card bg-gray-800 rounded-lg border border-gray-700 overflow-hidden">
                        <div className="p-4 border-b border-gray-700 flex justify-between items-center">
                            <h3 className="font-bold text-white">Archivos y Versiones</h3>
                            <div>
                                <label className="cursor-pointer bg-gray-700 hover:bg-gray-600 text-white text-sm px-3 py-1.5 rounded inline-block">
                                    {uploading ? 'Subiendo...' : '+ Subir Nueva Versión'}
                                    <input type="file" className="hidden" onChange={handleUpload} disabled={uploading} />
                                </label>
                            </div>
                        </div>
                        <table className="w-full text-left text-sm">
                            <thead>
                                <tr className="text-gray-500 border-b border-gray-700">
                                    <th className="p-3">Ver</th>
                                    <th className="p-3">Nombre Archivo</th>
                                    <th className="p-3">Tamaño</th>
                                    <th className="p-3">Subido</th>
                                    <th className="p-3 text-right">Acción</th>
                                </tr>
                            </thead>
                            <tbody>
                                {document.files && document.files.map((file: any) => (
                                    <tr key={file.id} className="border-b border-gray-800 hover:bg-gray-750">
                                        <td className="p-3 text-gray-400">v{file.version}</td>
                                        <td className="p-3 text-white">{file.file_name}</td>
                                        <td className="p-3 text-gray-400">{(file.size_bytes / 1024).toFixed(1)} KB</td>
                                        <td className="p-3 text-gray-500">{new Date(file.created_at).toLocaleDateString()}</td>
                                        <td className="p-3 text-right">
                                            <button
                                                onClick={() => handleDownload(file.version)}
                                                className="text-blue-400 hover:text-blue-300"
                                            >
                                                Descargar
                                            </button>
                                        </td>
                                    </tr>
                                ))}
                                {(!document.files || document.files.length === 0) && (
                                    <tr><td colSpan={5} className="p-6 text-center text-gray-500">No hay archivos subidos.</td></tr>
                                )}
                            </tbody>
                        </table>
                    </div>

                    {/* HISTORY */}
                    <div className="card bg-gray-800 rounded-lg border border-gray-700 overflow-hidden">
                        <div className="p-4 border-b border-gray-700">
                            <h3 className="font-bold text-white">Historial de Envíos</h3>
                        </div>
                        <div className="p-4">
                            {document.sends && document.sends.length > 0 ? (
                                <ul className="space-y-4">
                                    {document.sends.map((send: any) => (
                                        <li key={send.id} className="p-3 bg-gray-900 rounded border border-gray-800">
                                            <div className="flex justify-between mb-1">
                                                <span className="text-white font-medium">{send.subject}</span>
                                                <span className={`text-xs px-2 py-0.5 rounded ${send.status === 'sent' ? 'bg-green-900 text-green-300' :
                                                    send.status === 'failed' ? 'bg-red-900 text-red-300' : 'bg-yellow-900 text-yellow-300'
                                                    }`}>
                                                    {send.status.toUpperCase()}
                                                </span>
                                            </div>
                                            <div className="text-xs text-gray-400">
                                                Para: {send.to_json?.join(', ')}
                                            </div>
                                            <div className="text-xs text-gray-500 mt-1">
                                                {new Date(send.created_at).toLocaleString()}
                                            </div>
                                            {send.error_message && (
                                                <div className="text-xs text-red-400 mt-1">Error: {send.error_message}</div>
                                            )}
                                        </li>
                                    ))}
                                </ul>
                            ) : (
                                <p className="text-gray-500 text-sm">Aún no se han enviado correos.</p>
                            )}
                        </div>
                    </div>
                </div>

                {/* SIDEBAR */}
                <div className="space-y-6">
                    <div className="card bg-gray-800 p-4 rounded-lg border border-gray-700">
                        <h4 className="text-gray-400 uppercase text-xs font-bold mb-3">Detalles</h4>
                        <div className="space-y-3 text-sm">
                            <div>
                                <span className="text-gray-500 block">Cliente</span>
                                <span className="text-white">{document.client ? document.client.name : '-'}</span>
                            </div>
                            <div>
                                <span className="text-gray-500 block">Creado Por</span>
                                <span className="text-white">{document.creator ? document.creator.name : 'Desconocido'}</span>
                            </div>
                            <div>
                                <span className="text-gray-500 block">Última Actualización</span>
                                <span className="text-white">{new Date(document.updated_at).toLocaleString()}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
