import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { Link, useNavigate } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

export default function DocumentsIndex() {
    const [documents, setDocuments] = useState([]);
    const [loading, setLoading] = useState(true);
    const navigate = useNavigate();

    useEffect(() => {
        loadDocuments();
    }, []);

    const loadDocuments = () => {
        axios.get('/api/documents')
            .then(res => {
                setDocuments(res.data.data); // Pagination data.data
                setLoading(false);
            })
            .catch(err => {
                console.error(err);
                setLoading(false);
            });
    };

    const handleCreate = async () => {
        const title = prompt("Ingrese el título del documento:");
        if (!title) return;

        try {
            const res = await axios.post('/api/documents', {
                title,
                type: 'other',
                status: 'draft',
                client_id: null // optional
            });
            // Redirect to detail
            navigate(`/app/documents/${res.data.id}`);
        } catch (error) {
            alert('Error al crear el documento');
        }
    };

    return (
        <div className="p-6">
            <PageHeader
                title="Documentos"
                subtitle="Gestión de contratos, propuestas y archivos"
                actions={
                    <button onClick={handleCreate} className="btn btn-primary">
                        + Nuevo Documento
                    </button>
                }
            />

            {loading ? (
                <div className="text-center p-10">Cargando...</div>
            ) : (
                <div className="card mt-6">
                    <div className="overflow-x-auto">
                        <table className="w-full text-left border-collapse">
                            <thead>
                                <tr className="border-b border-gray-700 text-gray-400">
                                    <th className="p-4">Título</th>
                                    <th className="p-4">Tipo</th>
                                    <th className="p-4">Estado</th>
                                    <th className="p-4">Creado</th>
                                    <th className="p-4 text-right">Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {documents.length === 0 ? (
                                    <tr><td colSpan={5} className="p-4 text-center text-gray-500">No se encontraron documentos.</td></tr>
                                ) : (
                                    documents.map((doc: any) => (
                                        <tr key={doc.id} className="border-b border-gray-800 hover:bg-gray-800/50">
                                            <td className="p-4 font-medium text-white">
                                                <Link to={`/app/documents/${doc.id}`} className="hover:text-blue-400">
                                                    {doc.title}
                                                </Link>
                                                {doc.latest_file && (
                                                    <span className="ml-2 text-xs bg-gray-700 px-2 py-1 rounded text-gray-300">
                                                        Adjunto
                                                    </span>
                                                )}
                                            </td>
                                            <td className="p-4 capitalize text-gray-300">{doc.type}</td>
                                            <td className="p-4">
                                                <span className={`px-2 py-1 rounded text-xs font-bold ${doc.status === 'sent' ? 'bg-green-500/20 text-green-400' :
                                                        doc.status === 'ready' ? 'bg-blue-500/20 text-blue-400' :
                                                            'bg-gray-600/20 text-gray-400'
                                                    }`}>
                                                    {doc.status}
                                                </span>
                                            </td>
                                            <td className="p-4 text-sm text-gray-400">
                                                {new Date(doc.created_at).toLocaleDateString()}
                                            </td>
                                            <td className="p-4 text-right">
                                                <Link to={`/app/documents/${doc.id}`} className="text-blue-400 hover:text-blue-300 mr-3">
                                                    Ver
                                                </Link>
                                            </td>
                                        </tr>
                                    ))
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            )}
        </div>
    );
}
