import { useState, useEffect, CSSProperties } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

/**
 * 🌐 PÁGINA: DOMINIOS - DASHBOARD
 * 
 * ✅ ESTILO ACTUALIZADO - Mismo diseño premium que Marketing
 */

export default function DomainsDashboard() {
    const [domains, setDomains] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        fetchDomains();
    }, []);

    const fetchDomains = async () => {
        try {
            const res = await axios.get('/api/domains');
            setDomains(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    // Calculate metrics
    const activeDomains = domains.filter(d => d.status === 'active').length;
    const expiringDomains = domains.filter(d => {
        const daysLeft = Math.ceil((new Date(d.expires_at).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
        return daysLeft <= 30 && daysLeft > 0;
    }).length;
    const totalDomains = domains.length;

    const metricCards = [
        {
            icon: '🌐',
            label: 'Dominios Activos',
            value: activeDomains.toString(),
            subtext: `${totalDomains} totales`,
            gradient: 'linear-gradient(135deg, #06b6d4, #0891b2)',
            glow: 'rgba(6, 182, 212, 0.4)',
        },
        {
            icon: '⚠️',
            label: 'Próximos a Vencer',
            value: expiringDomains.toString(),
            subtext: 'En los próximos 30 días',
            gradient: 'linear-gradient(135deg, #f59e0b, #d97706)',
            glow: 'rgba(245, 158, 11, 0.4)',
        },
        {
            icon: '🔐',
            label: 'Con SSL',
            value: domains.filter(d => d.has_ssl).length.toString(),
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando dominios...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Gestión de Dominios"
                subtitle="Administra tus dominios, DNS y certificados SSL"
                actions={
                    <Link to="/app/domains" className="btn btn-primary">
                        🌐 Ver Dominios
                    </Link>
                }
                stats={[
                    { icon: '📅', label: 'Hoy', value: new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' }) },
                    { icon: '📊', label: 'Total dominios', value: totalDomains },
                ]}
            />

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Domains Table */}
            <div style={{ marginTop: '2.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🌐 Dominios Recientes</h2>
                        <Link to="/app/domains" className="btn btn-ghost btn-sm">
                            Ver todos →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Dominio</th>
                                    <th>Registrador</th>
                                    <th>Estado</th>
                                    <th>Vencimiento</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {domains.length === 0 ? (
                                    <tr>
                                        <td colSpan={5} style={{ textAlign: 'center', padding: '3rem', color: 'var(--text-muted)' }}>
                                            <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>🌐</div>
                                            <div>No hay dominios registrados</div>
                                        </td>
                                    </tr>
                                ) : (
                                    domains.slice(0, 5).map((domain) => {
                                        const daysLeft = Math.ceil((new Date(domain.expires_at).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
                                        const isExpiring = daysLeft <= 30 && daysLeft > 0;

                                        return (
                                            <tr key={domain.id} className={isExpiring ? 'bg-orange-500/10' : ''}>
                                                <td>
                                                    <div style={{ fontWeight: 600 }}>{domain.fqdn}</div>
                                                    {domain.has_ssl && <span className="badge badge-success text-xs">🔐 SSL</span>}
                                                </td>
                                                <td>{domain.registrar}</td>
                                                <td>
                                                    <span className={`badge ${domain.status === 'active' ? 'badge-success' :
                                                            domain.status === 'expired' ? 'badge-danger' :
                                                                'badge-warning'
                                                        }`}>
                                                        {domain.status}
                                                    </span>
                                                </td>
                                                <td>
                                                    <div className={daysLeft <= 7 ? 'text-red-400 font-bold' : ''}>
                                                        {domain.expires_at}
                                                    </div>
                                                    {isExpiring && (
                                                        <div style={{ fontSize: '0.75rem', color: '#f59e0b' }}>⚠️ {daysLeft} días</div>
                                                    )}
                                                </td>
                                                <td>
                                                    <Link to={`/app/domains/${domain.id}`} className="btn btn-sm btn-ghost">
                                                        Ver →
                                                    </Link>
                                                </td>
                                            </tr>
                                        );
                                    })
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    );
}
