import { useState, useEffect } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 MÓDULO: DOMINIOS - LISTADO
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el diseño premium estándar
 */

interface Domain {
    id: number;
    fqdn: string;
    client_id?: number;
    registrar_name?: string;
    status: 'active' | 'expired' | 'pending_transfer' | 'cancelled';
    registered_at?: string;
    expires_at: string;
    auto_renew: boolean;
    notes?: string;
}

const statusConfig: Record<string, { label: string; class: string; emoji: string }> = {
    active: { label: 'Activo', class: 'badge-success', emoji: '✅' },
    expired: { label: 'Vencido', class: 'badge-danger', emoji: '❌' },
    pending_transfer: { label: 'En Transferencia', class: 'badge-warning', emoji: '🔄' },
    cancelled: { label: 'Cancelado', class: 'badge-default', emoji: '⏹️' },
};

export default function DomainsIndex() {
    const [domains, setDomains] = useState<Domain[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [filter, setFilter] = useState<'all' | 'expiring'>('all');

    // Form State
    const [formData, setFormData] = useState({
        fqdn: '',
        registrar_name: '',
        status: 'active',
        expires_at: '',
        auto_renew: false,
    });

    useEffect(() => {
        fetchDomains();
    }, [filter]);

    const fetchDomains = async () => {
        try {
            const params = filter === 'expiring' ? { expiring_days: 30 } : {};
            const res = await axios.get('/api/domains', { params });
            setDomains(res.data);
        } catch (error) {
            console.error("Error cargando dominios:", error);
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            await axios.post('/api/domains', formData);
            setShowModal(false);
            fetchDomains();
            setFormData({
                fqdn: '',
                registrar_name: '',
                status: 'active',
                expires_at: '',
                auto_renew: false,
            });
        } catch (error) {
            console.error(error);
            alert('Error al crear el dominio');
        }
    };

    const getDaysUntilExpiration = (expiresAt: string) => {
        const days = Math.ceil((new Date(expiresAt).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
        return days;
    };

    const stats = [
        { icon: '🌐', label: 'Total Dominios', value: domains.length },
        { icon: '✅', label: 'Activos', value: domains.filter(d => d.status === 'active').length },
        { icon: '⚠️', label: 'Vencen <30 días', value: domains.filter(d => getDaysUntilExpiration(d.expires_at) <= 30 && getDaysUntilExpiration(d.expires_at) > 0).length },
    ];

    if (loading) return <div className="p-8 text-center text-slate-400">Cargando dominios...</div>;

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Gestión de Dominios"
                subtitle="Inventario, DNS, credenciales y recordatorios"
                actions={
                    <button className="btn btn-primary" onClick={() => setShowModal(true)}>
                        ➕ Nuevo Dominio
                    </button>
                }
                stats={stats}
            />

            {/* Filtros */}
            <div className="flex gap-2 mb-4">
                <button
                    className={`btn ${filter === 'all' ? 'btn-primary' : 'btn-ghost'}`}
                    onClick={() => setFilter('all')}
                >
                    Todos
                </button>
                <button
                    className={`btn ${filter === 'expiring' ? 'btn-warning' : 'btn-ghost'}`}
                    onClick={() => setFilter('expiring')}
                >
                    ⚠️ Próximos a Vencer
                </button>
            </div>

            <div className="table-container">
                <div className="table-header">
                    <h2 className="table-title">🌐 Dominios Registrados</h2>
                </div>
                <div className="table-wrapper">
                    <table className="table">
                        <thead>
                            <tr>
                                <th>Dominio</th>
                                <th>Registrador</th>
                                <th>Estado</th>
                                <th>Vence</th>
                                <th>Auto-Renovación</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            {domains.length === 0 ? (
                                <tr>
                                    <td colSpan={6} className="text-center p-8 text-slate-500">
                                        No hay dominios registrados
                                    </td>
                                </tr>
                            ) : (
                                domains.map((domain) => {
                                    const daysLeft = getDaysUntilExpiration(domain.expires_at);
                                    const isExpiring = daysLeft <= 30 && daysLeft > 0;

                                    return (
                                        <tr key={domain.id} className={isExpiring ? 'bg-orange-500/10' : ''}>
                                            <td>
                                                <div className="font-bold text-white">{domain.fqdn}</div>
                                                {domain.notes && <div className="text-xs text-slate-400">{domain.notes}</div>}
                                            </td>
                                            <td className="text-sm">{domain.registrar_name || 'N/A'}</td>
                                            <td>
                                                <span className={`badge ${statusConfig[domain.status]?.class || 'badge-default'}`}>
                                                    {statusConfig[domain.status]?.emoji} {statusConfig[domain.status]?.label}
                                                </span>
                                            </td>
                                            <td>
                                                <div className={daysLeft <= 7 ? 'text-red-400 font-bold' : ''}>
                                                    {domain.expires_at}
                                                </div>
                                                {isExpiring && (
                                                    <div className="text-xs text-orange-400">⚠️ {daysLeft} días</div>
                                                )}
                                            </td>
                                            <td>
                                                {domain.auto_renew ? (
                                                    <span className="text-green-400">✅ Sí</span>
                                                ) : (
                                                    <span className="text-slate-500">❌ No</span>
                                                )}
                                            </td>
                                            <td>
                                                <Link to={`/app/domains/${domain.id}`} className="btn btn-sm btn-ghost hover:text-blue-400">
                                                    🔧 Gestionar
                                                </Link>
                                            </td>
                                        </tr>
                                    );
                                })
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Modal Crear Dominio - Estilo Marketing */}
            {showModal && (
                <div style={{
                    position: 'fixed',
                    inset: 0,
                    background: 'rgba(0, 0, 0, 0.7)',
                    backdropFilter: 'blur(4px)',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    zIndex: 9999,
                    padding: '1rem'
                }} onClick={() => setShowModal(false)}>
                    <div className="card" style={{
                        maxWidth: '600px',
                        width: '100%',
                        maxHeight: '90vh',
                        overflowY: 'auto'
                    }} onClick={(e) => e.stopPropagation()}>
                        <div style={{ padding: '1.5rem', borderBottom: '1px solid var(--border-color)' }}>
                            <h2 style={{ fontSize: '1.5rem', fontWeight: 700, color: 'var(--text-primary)' }}>
                                🌐 Nuevo Dominio
                            </h2>
                            <p style={{ fontSize: '0.9rem', color: 'var(--text-muted)', marginTop: '0.5rem' }}>
                                Registra un nuevo dominio en el sistema
                            </p>
                        </div>
                        <form onSubmit={handleSubmit} style={{ padding: '1.5rem' }}>
                            <div style={{ display: 'flex', flexDirection: 'column', gap: '1.25rem' }}>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Dominio (FQDN)
                                    </label>
                                    <input
                                        type="text"
                                        required
                                        className="input"
                                        placeholder="ejemplo.com"
                                        value={formData.fqdn}
                                        onChange={(e) => setFormData({ ...formData, fqdn: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Registrador
                                    </label>
                                    <input
                                        type="text"
                                        className="input"
                                        placeholder="GoDaddy, Namecheap..."
                                        value={formData.registrar_name}
                                        onChange={(e) => setFormData({ ...formData, registrar_name: e.target.value })}
                                    />
                                </div>
                                <div>
                                    <label style={{ display: 'block', marginBottom: '0.5rem', fontWeight: 500 }}>
                                        Fecha de vencimiento
                                    </label>
                                    <input
                                        type="date"
                                        required
                                        className="input"
                                        value={formData.expires_at}
                                        onChange={(e) => setFormData({ ...formData, expires_at: e.target.value })}
                                    />
                                </div>
                                <div style={{ display: 'flex', alignItems: 'center', gap: '0.5rem' }}>
                                    <input
                                        type="checkbox"
                                        id="auto_renew"
                                        checked={formData.auto_renew}
                                        onChange={(e) => setFormData({ ...formData, auto_renew: e.target.checked })}
                                    />
                                    <label htmlFor="auto_renew" style={{ fontSize: '0.875rem' }}>Auto-renovación activada</label>
                                </div>
                                <div style={{ display: 'flex', gap: '0.75rem', justifyContent: 'flex-end', marginTop: '1rem' }}>
                                    <button type="button" className="btn btn-secondary" onClick={() => setShowModal(false)}>
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        ✨ Crear Dominio
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
