import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import axios from 'axios';
import PageHeader from '../../components/PageHeader';

/**
 * 🎨 MÓDULO: DOMINIOS - DETALLE
 * 
 * ✅ ESTILO ACTUALIZADO - Usa el diseño premium estándar
 */

export default function DomainsShow() {
    const { id } = useParams();
    const [domain, setDomain] = useState<any>(null);
    const [activeTab, setActiveTab] = useState<'overview' | 'dns' | 'vault' | 'reminders'>('overview');
    const [loading, setLoading] = useState(true);

    // Vault State
    const [showVaultModal, setShowVaultModal] = useState(false);
    const [revealingItem, setRevealingItem] = useState<any>(null);
    const [pin, setPin] = useState('');
    const [revealedSecret, setRevealedSecret] = useState('');
    const [secretTimer, setSecretTimer] = useState(0);

    useEffect(() => {
        fetchDomain();
    }, [id]);

    // Timer para ocultar secreto después de 30s
    useEffect(() => {
        if (secretTimer > 0) {
            const interval = setInterval(() => {
                setSecretTimer(prev => {
                    if (prev <= 1) {
                        setRevealedSecret('');
                        return 0;
                    }
                    return prev - 1;
                });
            }, 1000);
            return () => clearInterval(interval);
        }
    }, [secretTimer]);

    const fetchDomain = async () => {
        try {
            const res = await axios.get(`/api/domains/${id}`);
            setDomain(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    const handleRevealSecret = async (e: React.FormEvent) => {
        e.preventDefault();
        try {
            const res = await axios.post(`/api/domains/vault-items/${revealingItem.id}/reveal`, { pin });
            setRevealedSecret(res.data.secret);
            setSecretTimer(30); // 30 segundos
            setPin('');
        } catch (error: any) {
            if (error.response?.status === 422) {
                alert('PIN incorrecto');
            } else if (error.response?.status === 429) {
                alert(error.response.data.message);
            } else {
                alert('Error al revelar secreto');
            }
        }
    };

    if (loading) return <div className="p-8 text-center text-slate-400">Cargando dominio...</div>;
    if (!domain) return <div className="p-8 text-center text-red-400">Dominio no encontrado</div>;

    const tabs = [
        { id: 'overview', label: '📊 Resumen', icon: '📊' },
        { id: 'dns', label: '🌐 DNS Records', icon: '🌐' },
        { id: 'vault', label: '🔐 Credenciales', icon: '🔐' },
        { id: 'reminders', label: '🔔 Recordatorios', icon: '🔔' },
    ];

    return (
        <div className="animate-fadeIn">
            <Link to="/app/domains" className="text-sm text-slate-400 hover:text-white mb-4 inline-block">← Volver a dominios</Link>

            <PageHeader
                title={domain.fqdn}
                subtitle={`Vence: ${domain.expires_at}`}
                actions={
                    <button className="btn btn-secondary">✏️ Editar</button>
                }
            />

            {/* Tabs */}
            <div className="flex border-b border-slate-700 mb-6">
                {tabs.map(tab => (
                    <button
                        key={tab.id}
                        onClick={() => setActiveTab(tab.id as any)}
                        className={`px-6 py-3 font-medium transition-colors border-b-2 ${activeTab === tab.id
                                ? 'border-primary-500 text-white bg-slate-800/50'
                                : 'border-transparent text-slate-400 hover:text-slate-200'
                            }`}
                    >
                        {tab.label}
                    </button>
                ))}
            </div>

            {/* Tab Content */}
            <div className="min-h-[300px]">
                {activeTab === 'overview' && (
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="card">
                            <h3 className="text-lg font-bold mb-4 text-white">Información General</h3>
                            <div className="space-y-3 text-sm">
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Registrador:</span>
                                    <span>{domain.registrar_name || 'N/A'}</span>
                                </div>
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Estado:</span>
                                    <span className="badge badge-success">{domain.status}</span>
                                </div>
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Registrado:</span>
                                    <span>{domain.registered_at || 'N/A'}</span>
                                </div>
                                <div className="flex justify-between border-b border-slate-700 pb-2">
                                    <span className="text-slate-400">Auto-Renovación:</span>
                                    <span>{domain.auto_renew ? '✅ Sí' : '❌ No'}</span>
                                </div>
                            </div>
                        </div>

                        <div className="card">
                            <h3 className="text-lg font-bold mb-4 text-white">Nameservers</h3>
                            {domain.nameservers && domain.nameservers.length > 0 ? (
                                <ul className="space-y-2">
                                    {domain.nameservers.map((ns: any) => (
                                        <li key={ns.id} className="text-sm text-slate-300 flex items-center gap-2">
                                            {ns.is_active ? '✅' : '❌'} {ns.host}
                                        </li>
                                    ))}
                                </ul>
                            ) : (
                                <p className="text-slate-500 text-sm">No hay nameservers configurados</p>
                            )}
                        </div>
                    </div>
                )}

                {activeTab === 'dns' && (
                    <div className="table-container">
                        <div className="table-header flex justify-between">
                            <h3 className="table-title">Registros DNS</h3>
                            <button className="btn btn-sm btn-primary">➕ Nuevo Registro</button>
                        </div>
                        <div className="table-wrapper">
                            <table className="table">
                                <thead>
                                    <tr>
                                        <th>Tipo</th>
                                        <th>Nombre</th>
                                        <th>Valor</th>
                                        <th>TTL</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {domain.dns_records && domain.dns_records.length > 0 ? domain.dns_records.map((record: any) => (
                                        <tr key={record.id}>
                                            <td><span className="badge badge-primary">{record.type}</span></td>
                                            <td className="font-mono text-sm">{record.name}</td>
                                            <td className="font-mono text-xs text-slate-400">{record.value}</td>
                                            <td>{record.ttl}s</td>
                                            <td><button className="text-slate-400 hover:text-white">✏️</button></td>
                                        </tr>
                                    )) : (
                                        <tr><td colSpan={5} className="text-center p-4 text-slate-500">No hay registros DNS</td></tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                )}

                {activeTab === 'vault' && (
                    <div className="table-container">
                        <div className="table-header flex justify-between">
                            <h3 className="table-title">🔐 Credenciales Cifradas</h3>
                            <button className="btn btn-sm btn-primary">➕ Nueva Credencial</button>
                        </div>
                        <div className="table-wrapper">
                            <table className="table">
                                <thead>
                                    <tr>
                                        <th>Proveedor</th>
                                        <th>Etiqueta</th>
                                        <th>Usuario</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {domain.vault_items && domain.vault_items.length > 0 ? domain.vault_items.map((item: any) => (
                                        <tr key={item.id}>
                                            <td><span className="badge badge-purple">{item.provider}</span></td>
                                            <td className="font-bold text-white">{item.label}</td>
                                            <td className="text-sm">{item.username || 'N/A'}</td>
                                            <td>
                                                <button
                                                    className="btn btn-sm btn-ghost hover:text-blue-400"
                                                    onClick={() => {
                                                        setRevealingItem(item);
                                                        setShowVaultModal(true);
                                                    }}
                                                >
                                                    👁️ Revelar
                                                </button>
                                            </td>
                                        </tr>
                                    )) : (
                                        <tr><td colSpan={4} className="text-center p-4 text-slate-500">No hay credenciales guardadas</td></tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                )}

                {activeTab === 'reminders' && (
                    <div className="card">
                        <h3 className="text-lg font-bold mb-4">Recordatorios Configurados</h3>
                        {domain.reminders && domain.reminders.length > 0 ? (
                            <div className="space-y-3">
                                {domain.reminders.map((reminder: any) => (
                                    <div key={reminder.id} className="p-4 bg-slate-800/50 rounded border border-slate-700">
                                        <div className="flex justify-between items-center">
                                            <div>
                                                <div className="font-bold">{reminder.kind}</div>
                                                <div className="text-sm text-slate-400">
                                                    Días: {reminder.schedule_days_json?.join(', ') || 'N/A'}
                                                </div>
                                            </div>
                                            <span className={`badge ${reminder.enabled ? 'badge-success' : 'badge-default'}`}>
                                                {reminder.enabled ? '✅ Activo' : '❌ Inactivo'}
                                            </span>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        ) : (
                            <p className="text-slate-500">No hay recordatorios configurados</p>
                        )}
                    </div>
                )}
            </div>

            {/* Modal Revelar Secreto */}
            {showVaultModal && revealingItem && (
                <div className="fixed inset-0 bg-black/70 backdrop-blur-sm z-50 flex items-center justify-center p-4">
                    <div className="card w-full max-w-md shadow-2xl border border-slate-700">
                        <div className="flex justify-between items-center border-b border-slate-700 pb-4 mb-4">
                            <h3 className="text-xl font-bold text-white">🔐 Revelar Credencial</h3>
                            <button onClick={() => {
                                setShowVaultModal(false);
                                setRevealedSecret('');
                                setPin('');
                            }} className="text-slate-400 hover:text-white">✕</button>
                        </div>

                        <div className="mb-4">
                            <p className="text-sm text-slate-400">Proveedor: <span className="text-white">{revealingItem.provider}</span></p>
                            <p className="text-sm text-slate-400">Etiqueta: <span className="text-white">{revealingItem.label}</span></p>
                        </div>

                        {!revealedSecret ? (
                            <form onSubmit={handleRevealSecret} className="space-y-4">
                                <div>
                                    <label className="block text-sm font-medium text-slate-300 mb-1">PIN de Seguridad (4 dígitos)</label>
                                    <input
                                        type="password"
                                        maxLength={4}
                                        pattern="\d{4}"
                                        required
                                        className="input w-full text-center text-2xl tracking-widest"
                                        placeholder="••••"
                                        value={pin}
                                        onChange={(e) => setPin(e.target.value.replace(/\D/g, ''))}
                                        autoFocus
                                    />
                                    <p className="text-xs text-slate-500 mt-1">Este PIN se configura en tu perfil de usuario</p>
                                </div>

                                <div className="flex justify-end gap-3">
                                    <button type="button" onClick={() => setShowVaultModal(false)} className="btn btn-ghost">
                                        Cancelar
                                    </button>
                                    <button type="submit" className="btn btn-primary">
                                        🔓 Revelar
                                    </button>
                                </div>
                            </form>
                        ) : (
                            <div className="space-y-4">
                                <div className="p-4 bg-green-500/10 border border-green-500/30 rounded">
                                    <p className="text-xs text-slate-400 mb-2">Contraseña revelada:</p>
                                    <p className="font-mono text-lg text-green-400 break-all select-all">{revealedSecret}</p>
                                </div>
                                <div className="text-center">
                                    <p className="text-sm text-orange-400">
                                        ⏱️ Se ocultará automáticamente en {secretTimer} segundos
                                    </p>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            )}
        </div>
    );
}
