import { useState, useEffect, CSSProperties } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import PageHeader from '../../components/PageHeader';

/**
 * 🖥️ PÁGINA: HOSTING - DASHBOARD
 * 
 * ✅ ESTILO ACTUALIZADO - Mismo diseño premium que Marketing
 */

export default function HostingDashboard() {
    const [services, setServices] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        fetchServices();
    }, []);

    const fetchServices = async () => {
        try {
            const res = await axios.get('/api/hosting/services');
            setServices(res.data);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    // Calculate metrics
    const activeServices = services.filter(s => s.status === 'active').length;
    const expiringServices = services.filter(s => {
        const daysLeft = Math.ceil((new Date(s.next_due_at).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
        return daysLeft <= 30 && daysLeft > 0;
    }).length;
    const totalRevenue = services.reduce((acc, s) => acc + Number(s.price || 0), 0);

    const metricCards = [
        {
            icon: '🖥️',
            label: 'Servicios Activos',
            value: activeServices.toString(),
            subtext: `${services.length} totales`,
            gradient: 'linear-gradient(135deg, #8b5cf6, #7c3aed)',
            glow: 'rgba(139, 92, 246, 0.4)',
        },
        {
            icon: '💰',
            label: 'Ingresos Mensuales',
            value: `$${totalRevenue.toLocaleString('es-ES', { minimumFractionDigits: 2 })}`,
            gradient: 'linear-gradient(135deg, #10b981, #059669)',
            glow: 'rgba(16, 185, 129, 0.4)',
        },
        {
            icon: '⚠️',
            label: 'Próximos a Vencer',
            value: expiringServices.toString(),
            subtext: 'En los próximos 30 días',
            gradient: 'linear-gradient(135deg, #f59e0b, #d97706)',
            glow: 'rgba(245, 158, 11, 0.4)',
        },
    ];

    if (loading) {
        return (
            <div className="animate-fadeIn" style={{ display: 'flex', alignItems: 'center', justifyContent: 'center', minHeight: '400px' }}>
                <div style={{ textAlign: 'center', color: 'var(--text-muted)' }}>
                    <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>⏳</div>
                    <div>Cargando servicios...</div>
                </div>
            </div>
        );
    }

    return (
        <div className="animate-fadeIn">
            <PageHeader
                title="Gestión de Hosting"
                subtitle="Administra tus servicios de hosting y servidores"
                actions={
                    <>
                        <Link to="/app/hosting/plans" className="btn btn-secondary mr-2">
                            📋 Planes
                        </Link>
                        <Link to="/app/hosting/services" className="btn btn-primary">
                            🖥️ Ver Servicios
                        </Link>
                    </>
                }
                stats={[
                    { icon: '📅', label: 'Hoy', value: new Date().toLocaleDateString('es-ES', { weekday: 'long', day: 'numeric', month: 'short' }) },
                    { icon: '📊', label: 'Total servicios', value: services.length },
                ]}
            />

            {/* Metric Cards Grid */}
            <div className="cards-grid">
                {metricCards.map((card, index) => (
                    <div
                        key={index}
                        className="card metric-card"
                        style={{ '--card-glow': card.glow } as CSSProperties}
                    >
                        <div
                            className="metric-icon"
                            style={{ background: card.gradient }}
                        >
                            {card.icon}
                        </div>
                        <div className="metric-content">
                            <div className="metric-label">{card.label}</div>
                            <div className="metric-value">{card.value}</div>
                            {card.subtext && (
                                <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)', marginTop: '0.25rem' }}>
                                    {card.subtext}
                                </div>
                            )}
                        </div>
                    </div>
                ))}
            </div>

            {/* Services Table */}
            <div style={{ marginTop: '2.5rem' }}>
                <div className="table-container">
                    <div className="table-header">
                        <h2 className="table-title">🖥️ Servicios Recientes</h2>
                        <Link to="/app/hosting/services" className="btn btn-ghost btn-sm">
                            Ver todos →
                        </Link>
                    </div>
                    <div className="table-wrapper">
                        <table className="table">
                            <thead>
                                <tr>
                                    <th>Servicio</th>
                                    <th>Plan</th>
                                    <th>Estado</th>
                                    <th>Vencimiento</th>
                                    <th>Precio</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                {services.length === 0 ? (
                                    <tr>
                                        <td colSpan={6} style={{ textAlign: 'center', padding: '3rem', color: 'var(--text-muted)' }}>
                                            <div style={{ fontSize: '2rem', marginBottom: '1rem' }}>🖥️</div>
                                            <div>No hay servicios registrados</div>
                                        </td>
                                    </tr>
                                ) : (
                                    services.slice(0, 5).map((service) => {
                                        const daysLeft = Math.ceil((new Date(service.next_due_at).getTime() - Date.now()) / (1000 * 60 * 60 * 24));
                                        const isExpiring = daysLeft <= 30 && daysLeft > 0;

                                        return (
                                            <tr key={service.id} className={isExpiring ? 'bg-orange-500/10' : ''}>
                                                <td>
                                                    <div style={{ fontWeight: 600 }}>{service.label}</div>
                                                    {service.domain_fqdn && <div style={{ fontSize: '0.75rem', color: 'var(--text-muted)' }}>{service.domain_fqdn}</div>}
                                                </td>
                                                <td>{service.plan?.name || 'N/A'}</td>
                                                <td>
                                                    <span className={`badge ${service.status === 'active' ? 'badge-success' :
                                                            service.status === 'suspended' ? 'badge-warning' :
                                                                'badge-danger'
                                                        }`}>
                                                        {service.status}
                                                    </span>
                                                </td>
                                                <td>
                                                    <div className={daysLeft <= 7 ? 'text-red-400 font-bold' : ''}>
                                                        {service.next_due_at}
                                                    </div>
                                                    {isExpiring && (
                                                        <div style={{ fontSize: '0.75rem', color: '#f59e0b' }}>⚠️ {daysLeft} días</div>
                                                    )}
                                                </td>
                                                <td style={{ fontFamily: 'monospace' }}>{service.currency} {service.price}</td>
                                                <td>
                                                    <Link to={`/app/hosting/services/${service.id}`} className="btn btn-sm btn-ghost">
                                                        Ver →
                                                    </Link>
                                                </td>
                                            </tr>
                                        );
                                    })
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    );
}
